<?php
require_once 'config.php';

$database = new Database();
$db = $database->getConnection();

echo "<h2>Migración de Base de Datos CRM</h2>";
echo "<pre>";

try {
    // Verificar si las nuevas columnas ya existen
    $check_estado = $db->query("SHOW COLUMNS FROM contactos LIKE 'estado'");
    $check_observaciones = $db->query("SHOW COLUMNS FROM contactos LIKE 'observaciones'");
    
    // Agregar columnas a la tabla contactos si no existen
    if ($check_estado->rowCount() == 0) {
        $db->exec("ALTER TABLE contactos ADD COLUMN estado ENUM('Frio', 'Tibio', 'Caliente') DEFAULT 'Frio'");
        echo "✓ Columna 'estado' agregada a la tabla contactos\n";
    } else {
        echo "✓ Columna 'estado' ya existe\n";
    }
    
    if ($check_observaciones->rowCount() == 0) {
        $db->exec("ALTER TABLE contactos ADD COLUMN observaciones TEXT");
        echo "✓ Columna 'observaciones' agregada a la tabla contactos\n";
    } else {
        echo "✓ Columna 'observaciones' ya existe\n";
    }
    
    // Crear tabla de etiquetas si no existe
    $create_etiquetas = "
    CREATE TABLE IF NOT EXISTS etiquetas (
        id INT AUTO_INCREMENT PRIMARY KEY,
        nombre VARCHAR(100) NOT NULL UNIQUE,
        color VARCHAR(7) DEFAULT '#007bff',
        fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )";
    $db->exec($create_etiquetas);
    echo "✓ Tabla 'etiquetas' creada/verificada\n";
    
    // Crear tabla de relación contactos-etiquetas si no existe
    $create_contacto_etiquetas = "
    CREATE TABLE IF NOT EXISTS contacto_etiquetas (
        id INT AUTO_INCREMENT PRIMARY KEY,
        contacto_id INT NOT NULL,
        etiqueta_id INT NOT NULL,
        fecha_asignacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (contacto_id) REFERENCES contactos(id) ON DELETE CASCADE,
        FOREIGN KEY (etiqueta_id) REFERENCES etiquetas(id) ON DELETE CASCADE,
        UNIQUE KEY unique_contacto_etiqueta (contacto_id, etiqueta_id)
    )";
    $db->exec($create_contacto_etiquetas);
    echo "✓ Tabla 'contacto_etiquetas' creada/verificada\n";
    
    // Crear tabla de historial de cambios si no existe
    $create_historial = "
    CREATE TABLE IF NOT EXISTS historial_cambios (
        id INT AUTO_INCREMENT PRIMARY KEY,
        contacto_id INT NOT NULL,
        usuario_id INT NOT NULL,
        campo_modificado VARCHAR(100) NOT NULL,
        valor_anterior TEXT,
        valor_nuevo TEXT,
        fecha_cambio TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (contacto_id) REFERENCES contactos(id) ON DELETE CASCADE,
        FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE
    )";
    $db->exec($create_historial);
    echo "✓ Tabla 'historial_cambios' creada/verificada\n";
    
    // Crear índices si no existen
    $indices = [
        "CREATE INDEX IF NOT EXISTS idx_estado ON contactos(estado)",
        "CREATE INDEX IF NOT EXISTS idx_contacto_etiquetas ON contacto_etiquetas(contacto_id)",
        "CREATE INDEX IF NOT EXISTS idx_historial_contacto ON historial_cambios(contacto_id)",
        "CREATE INDEX IF NOT EXISTS idx_historial_fecha ON historial_cambios(fecha_cambio)"
    ];
    
    foreach ($indices as $indice) {
        try {
            $db->exec($indice);
        } catch (Exception $e) {
            // Ignorar errores de índices que ya existen
        }
    }
    echo "✓ Índices creados/verificados\n";
    
    // Insertar etiquetas de ejemplo si no existen
    $check_etiquetas = $db->query("SELECT COUNT(*) FROM etiquetas")->fetchColumn();
    if ($check_etiquetas == 0) {
        $etiquetas_ejemplo = [
            ['Cliente VIP', '#28a745'],
            ['Prospecto', '#ffc107'],
            ['Seguimiento', '#17a2b8'],
            ['Urgente', '#dc3545'],
            ['Interesado', '#6f42c1']
        ];
        
        $stmt = $db->prepare("INSERT INTO etiquetas (nombre, color) VALUES (?, ?)");
        foreach ($etiquetas_ejemplo as $etiqueta) {
            $stmt->execute($etiqueta);
        }
        echo "✓ Etiquetas de ejemplo insertadas\n";
    } else {
        echo "✓ Etiquetas ya existen\n";
    }
    
    echo "\n🎉 Migración completada exitosamente!\n";
    echo "\nAhora puedes usar contactos.php sin errores.\n";
    
} catch (Exception $e) {
    echo "❌ Error durante la migración: " . $e->getMessage() . "\n";
    echo "\nDetalles del error:\n";
    echo $e->getTraceAsString();
}

echo "</pre>";
?>

<style>
body { font-family: Arial, sans-serif; margin: 20px; }
pre { background: #f5f5f5; padding: 15px; border-radius: 5px; }
</style>