<?php
require_once 'config.php';

// Verificar si el usuario está logueado
if (!isLoggedIn()) {
    header('Location: login.php');
    exit();
}

// Inicializar conexión a la base de datos
$database = new Database();
$db = $database->getConnection();

$mensaje = '';
$tipo_mensaje = '';
$resultados = [];

// Procesar la carga del archivo
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['excel_file'])) {
    $archivo = $_FILES['excel_file'];
    
    // Validar que se subió un archivo
    if ($archivo['error'] !== UPLOAD_ERR_OK) {
        $mensaje = 'Error al subir el archivo.';
        $tipo_mensaje = 'danger';
    } else {
        // Validar extensión del archivo
        $extension = strtolower(pathinfo($archivo['name'], PATHINFO_EXTENSION));
        if (!in_array($extension, ['xlsx', 'xls', 'csv'])) {
            $mensaje = 'Solo se permiten archivos Excel (.xlsx, .xls) o CSV (.csv)';
            $tipo_mensaje = 'danger';
        } else {
            try {
                $contactos_importados = 0;
                $contactos_duplicados = 0;
                $errores = [];
                $datos = [];
                
                // Verificar que el archivo temporal existe
                if (!file_exists($archivo['tmp_name'])) {
                    throw new Exception('El archivo temporal no existe');
                }
                
                // Procesar según el tipo de archivo
                if ($extension === 'csv') {
                    // Procesar archivo CSV
                    if (($handle = fopen($archivo['tmp_name'], 'r')) !== FALSE) {
                        $row_num = 0;
                        while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
                            $row_num++;
                            if ($row_num === 1) continue; // Saltar header
                            
                            $datos[] = [
                                'nombre' => isset($data[0]) ? trim($data[0]) : '',
                                'celular' => isset($data[1]) ? trim($data[1]) : '',
                                'ciudad' => isset($data[2]) ? trim($data[2]) : '',
                                'fecha' => isset($data[3]) ? trim($data[3]) : '',
                                'fila' => $row_num
                            ];
                        }
                        fclose($handle);
                    }
                } else {
                    // Procesar archivo Excel usando PhpSpreadsheet
                    try {
                        require_once 'vendor/autoload.php';
                        
                        $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReader(ucfirst($extension));
                        $reader->setReadDataOnly(true);
                        $spreadsheet = $reader->load($archivo['tmp_name']);
                        $worksheet = $spreadsheet->getActiveSheet();
                        $highestRow = $worksheet->getHighestRow();
                        
                        for ($row = 2; $row <= $highestRow; $row++) {
                            $nombre = '';
                            $celular = '';
                            $ciudad = '';
                            $fecha = '';
                            
                            try {
                                $nombre = trim((string)$worksheet->getCell('A' . $row)->getCalculatedValue());
                                $celular = trim((string)$worksheet->getCell('B' . $row)->getCalculatedValue());
                                $ciudad = trim((string)$worksheet->getCell('C' . $row)->getCalculatedValue());
                                $fecha = $worksheet->getCell('D' . $row)->getCalculatedValue();
                            } catch (Exception $e) {
                                // Si hay error leyendo una celda, continuar con valores vacíos
                            }
                            
                            $datos[] = [
                                'nombre' => $nombre,
                                'celular' => $celular,
                                'ciudad' => $ciudad,
                                'fecha' => $fecha,
                                'fila' => $row
                            ];
                        }
                    } catch (Exception $e) {
                        throw new Exception('Error al procesar archivo Excel: ' . $e->getMessage());
                    }
                }
                
                // Procesar los datos
                foreach ($datos as $fila_data) {
                    $nombre = $fila_data['nombre'];
                    $celular = $fila_data['celular'];
                    $ciudad = $fila_data['ciudad'];
                    $fecha_registro = $fila_data['fecha'];
                    $row = $fila_data['fila'];
                    
                    // Validar campos requeridos
                    if (empty($nombre) || empty($celular)) {
                        $errores[] = "Fila $row: Nombre y celular son obligatorios";
                        continue;
                    }
                    
                    // Validar formato de celular
                    $celular_limpio = preg_replace('/[^0-9]/', '', $celular);
                    if (strlen($celular_limpio) < 7) {
                        $errores[] = "Fila $row: Celular debe tener al menos 7 dígitos";
                        continue;
                    }
                    
                    // Formatear fecha
                    $fecha_formateada = date('Y-m-d');
                    if (!empty($fecha_registro)) {
                        try {
                            if (is_numeric($fecha_registro) && $extension !== 'csv') {
                                // Fecha de Excel (número serial)
                                $fecha_formateada = \PhpOffice\PhpSpreadsheet\Shared\Date::excelToDateTimeObject($fecha_registro)->format('Y-m-d');
                            } else {
                                // Fecha como texto
                                $fecha_obj = DateTime::createFromFormat('Y-m-d', $fecha_registro);
                                if (!$fecha_obj) {
                                    $fecha_obj = DateTime::createFromFormat('d/m/Y', $fecha_registro);
                                }
                                if (!$fecha_obj) {
                                    $fecha_obj = new DateTime($fecha_registro);
                                }
                                $fecha_formateada = $fecha_obj->format('Y-m-d');
                            }
                        } catch (Exception $e) {
                            $fecha_formateada = date('Y-m-d');
                        }
                    }
                    
                    // Verificar si el contacto ya existe
                    $stmt = $db->prepare("SELECT id, nombres_apellidos, usuario_id FROM contactos WHERE celular = ?");
                    $stmt->execute([$celular_limpio]);
                    $contacto_existente = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($contacto_existente) {
                        // Verificar si el contacto pertenece al mismo usuario
                        if ($contacto_existente['usuario_id'] == $_SESSION['user_id']) {
                            // Contacto del mismo usuario - actualizar fecha y registrar recontacto
                            $contactos_duplicados++;
                            
                            // Actualizar la fecha del contacto existente
                            try {
                                $stmt = $db->prepare("UPDATE contactos SET fecha_registro = ? WHERE id = ?");
                                $stmt->execute([$fecha_formateada, $contacto_existente['id']]);
                            } catch (Exception $e) {
                                // Error al actualizar fecha
                            }
                            
                            // Registrar en historial que el contacto volvió a contactar
                            try {
                                $stmt = $db->prepare("INSERT INTO historial_cambios (contacto_id, campo_modificado, valor_anterior, valor_nuevo, usuario_id, fecha_cambio) VALUES (?, ?, ?, ?, ?, NOW())");
                                $stmt->execute([
                                    $contacto_existente['id'],
                                    'recontacto',
                                    'Contacto anterior',
                                    'Contacto volvió a contactar el ' . date('d/m/Y', strtotime($fecha_formateada)),
                                    $_SESSION['user_id']
                                ]);
                            } catch (Exception $e) {
                                // Ignorar si no existe la tabla historial
                            }
                            
                            $resultados[] = [
                                'tipo' => 'recontacto',
                                'nombre' => $nombre,
                                'celular' => $celular_limpio,
                                'mensaje' => 'Contacto existente - Registrado recontacto el ' . date('d/m/Y', strtotime($fecha_formateada))
                            ];
                        } else {
                            // Contacto pertenece a otro usuario - no cargar, solo registrar en historial
                            $contactos_duplicados++;
                            
                            // Registrar en historial que se intentó cargar un contacto de otro usuario
                            try {
                                $stmt = $db->prepare("INSERT INTO historial_cambios (contacto_id, campo_modificado, valor_anterior, valor_nuevo, usuario_id, fecha_cambio) VALUES (?, ?, ?, ?, ?, NOW())");
                                $stmt->execute([
                                    null, // No hay contacto_id porque no se carga
                                    'intento_duplicacion',
                                    'Contacto de otro usuario',
                                    'Intento de carga de contacto existente: ' . $nombre . ' (' . $celular_limpio . ') el ' . date('d/m/Y', strtotime($fecha_formateada)),
                                    $_SESSION['user_id']
                                ]);
                            } catch (Exception $e) {
                                // Ignorar si no existe la tabla historial
                            }
                            
                            $resultados[] = [
                                'tipo' => 'otro_usuario',
                                'nombre' => $nombre,
                                'celular' => $celular_limpio,
                                'mensaje' => 'Contacto pertenece a otro usuario - no se puede cargar'
                            ];
                        }
                    } else {
                        // Insertar nuevo contacto
                        $stmt = $db->prepare("INSERT INTO contactos (nombres_apellidos, celular, ciudad, fecha_registro, estado, usuario_id) VALUES (?, ?, ?, ?, ?, ?)");
                        if ($stmt->execute([$nombre, $celular_limpio, $ciudad, $fecha_formateada, 'Frio', $_SESSION['user_id']])) {
                            $contactos_importados++;
                            $resultados[] = [
                                'tipo' => 'importado',
                                'nombre' => $nombre,
                                'celular' => $celular_limpio,
                                'mensaje' => 'Contacto importado exitosamente'
                            ];
                        } else {
                            $errores[] = "Fila $row: Error al insertar contacto";
                        }
                    }
                }
                
                // Contar diferentes tipos de resultados
                $recontactos = 0;
                $otros_usuarios = 0;
                foreach ($resultados as $resultado) {
                    if ($resultado['tipo'] === 'recontacto') {
                        $recontactos++;
                    } elseif ($resultado['tipo'] === 'otro_usuario') {
                        $otros_usuarios++;
                    }
                }
                
                // Mensaje de resultado
                if ($contactos_importados > 0 || $recontactos > 0 || $otros_usuarios > 0) {
                    $mensaje = "Procesamiento completado: $contactos_importados contactos nuevos";
                    if ($recontactos > 0) {
                        $mensaje .= ", $recontactos recontactos";
                    }
                    if ($otros_usuarios > 0) {
                        $mensaje .= ", $otros_usuarios contactos de otros usuarios (no cargados)";
                    }
                    if (count($errores) > 0) {
                        $mensaje .= ", " . count($errores) . " errores";
                    }
                    $tipo_mensaje = 'success';
                } else {
                    $mensaje = 'No se procesaron contactos. Revisa el formato del archivo.';
                    $tipo_mensaje = 'warning';
                }
                
            } catch (Exception $e) {
                $mensaje = 'Error al procesar el archivo: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Importar Contactos - CRM</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #dc2626;
            --secondary-color: #b91c1c;
            --accent-color: #f87171;
            --success-color: #16a34a;
            --warning-color: #ca8a04;
            --danger-color: #dc2626;
            --dark-color: #1f2937;
            --light-color: #f8fafc;
            --text-primary: #1f2937;
            --text-secondary: #6b7280;
            --header-height: 70px;
        }
        
        * {
            font-family: 'Poppins', sans-serif;
        }

        html, body {
            margin: 0;
            padding: 0;
        }

        body {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            color: var(--text-primary);
            min-height: 100vh;
            padding-top: var(--header-height);
        }
        
        /* Estilos específicos de importar excel */
        
        .main-content {
            padding: 2rem;
        }
        
        .import-card {
            background: white;
            border-radius: 15px;
            padding: 2rem;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            margin-bottom: 2rem;
        }
        
        .upload-area {
            border: 2px dashed #ddd;
            border-radius: 10px;
            padding: 3rem;
            text-align: center;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .upload-area:hover {
            border-color: var(--primary-color);
            background: rgba(102, 126, 234, 0.05);
        }
        
        .upload-area.dragover {
            border-color: var(--primary-color);
            background: rgba(102, 126, 234, 0.1);
        }
        
        .results-table {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }
        
        .table th {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border: none;
            font-weight: 600;
        }
        
        .badge-success {
            background: var(--success-color);
        }
        
        .badge-warning {
            background: var(--warning-color);
        }
        
        .badge-info {
            background: #17a2b8;
            color: white;
        }
        
        .badge-warning {
            background: #ffc107;
            color: #212529;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>

    <!-- Main Content -->
    <div class="main-content">
        <div class="container-fluid">
            <div class="row">
                <div class="col-12">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <div>
                            <h2 class="mb-1 text-white"><i class="fas fa-file-excel me-2"></i>Importar Contactos</h2>
                            <p class="text-white-50 mb-0">Carga masiva de contactos desde archivos Excel o CSV</p>
                        </div>
                        <a href="contactos.php" class="btn btn-outline-light">
                            <i class="fas fa-arrow-left me-2"></i>Volver a Contactos
                        </a>
                    </div>
                </div>
            </div>
            
            <?php if ($mensaje): ?>
            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show" role="alert">
                <i class="fas fa-<?php echo $tipo_mensaje === 'success' ? 'check-circle' : 'exclamation-triangle'; ?> me-2"></i>
                <?php echo $mensaje; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>
            
            <div class="row">
                <div class="col-lg-8">
                    <div class="import-card">
                        <h4 class="mb-4"><i class="fas fa-upload me-2"></i>Subir Archivo</h4>
                        
                        <form method="POST" enctype="multipart/form-data" id="uploadForm">
                            <div class="upload-area" id="uploadArea">
                                <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3"></i>
                                <h5>Arrastra tu archivo aquí o haz clic para seleccionar</h5>
                                <p class="text-muted mb-3">Formatos: .xlsx, .xls, .csv (Máximo 10MB)</p>
                                <input type="file" name="excel_file" id="excel_file" accept=".xlsx,.xls,.csv" class="d-none" required>
                                <button type="button" class="btn btn-primary" onclick="document.getElementById('excel_file').click()">
                                    <i class="fas fa-folder-open me-2"></i>Seleccionar Archivo
                                </button>
                            </div>
                            
                            <div class="mt-3" id="fileInfo" style="display: none;">
                                <div class="alert alert-info">
                                    <i class="fas fa-file me-2"></i>
                                    <span id="fileName"></span>
                                    <button type="button" class="btn btn-sm btn-outline-danger ms-2" onclick="clearFile()">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <div class="text-center mt-4">
                                <button type="submit" class="btn btn-primary btn-lg" id="submitBtn" disabled>
                                    <i class="fas fa-upload me-2" id="submitIcon"></i>
                                    <span id="submitText">Importar Contactos</span>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <div class="col-lg-4">
                    <div class="import-card">
                        <h5 class="mb-3"><i class="fas fa-info-circle me-2"></i>Formato del Archivo</h5>
                        <p class="text-muted mb-3">Tu archivo debe tener las siguientes columnas:</p>
                        
                        <div class="mb-3">
                            <a href="plantilla_contactos.csv" download class="btn btn-outline-success btn-sm">
                                <i class="fas fa-download me-2"></i>Descargar Plantilla CSV
                            </a>
                        </div>
                        
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Columna</th>
                                        <th>Campo</th>
                                        <th>Requerido</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td><strong>A/1</strong></td>
                                        <td>Nombre</td>
                                        <td><span class="badge bg-danger">Sí</span></td>
                                    </tr>
                                    <tr>
                                        <td><strong>B/2</strong></td>
                                        <td>Celular</td>
                                        <td><span class="badge bg-danger">Sí</span></td>
                                    </tr>
                                    <tr>
                                        <td><strong>C/3</strong></td>
                                        <td>Ciudad</td>
                                        <td><span class="badge bg-secondary">No</span></td>
                                    </tr>
                                    <tr>
                                        <td><strong>D/4</strong></td>
                                        <td>Fecha Registro</td>
                                        <td><span class="badge bg-secondary">No</span></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-3">
                            <small class="text-muted">
                                <strong>Nota:</strong> Los archivos CSV deben usar coma (,) como separador.
                            </small>
                        </div>
                    </div>
                </div>
            </div>
            
            <?php if (!empty($resultados)): ?>
            <div class="row mt-4">
                <div class="col-12">
                    <div class="results-table">
                        <h5 class="p-3 mb-0"><i class="fas fa-list me-2"></i>Resultados de la Importación</h5>
                        <div class="table-responsive">
                            <table class="table table-striped mb-0">
                                <thead>
                                    <tr>
                                        <th>Estado</th>
                                        <th>Nombre</th>
                                        <th>Celular</th>
                                        <th>Mensaje</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($resultados as $resultado): ?>
                                    <tr>
                                        <td>
                                            <?php if ($resultado['tipo'] === 'importado'): ?>
                                                <span class="badge badge-success">Importado</span>
                                            <?php elseif ($resultado['tipo'] === 'recontacto'): ?>
                                                <span class="badge badge-info">Recontacto</span>
                                            <?php elseif ($resultado['tipo'] === 'otro_usuario'): ?>
                                                <span class="badge badge-warning">Otro Usuario</span>
                                            <?php else: ?>
                                                <span class="badge badge-warning">Duplicado</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($resultado['nombre']); ?></td>
                                        <td><?php echo htmlspecialchars($resultado['celular']); ?></td>
                                        <td><?php echo htmlspecialchars($resultado['mensaje']); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($errores)): ?>
            <div class="row mt-4">
                <div class="col-12">
                    <div class="alert alert-warning">
                        <h6><i class="fas fa-exclamation-triangle me-2"></i>Errores encontrados:</h6>
                        <ul class="mb-0">
                            <?php foreach ($errores as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Manejo de archivos
        const fileInput = document.getElementById('excel_file');
        const fileInfo = document.getElementById('fileInfo');
        const fileName = document.getElementById('fileName');
        const submitBtn = document.getElementById('submitBtn');
        const uploadArea = document.getElementById('uploadArea');
        
        fileInput.addEventListener('change', function(e) {
            if (this.files && this.files.length > 0) {
                const file = this.files[0];
                fileName.textContent = file.name;
                fileInfo.style.display = 'block';
                submitBtn.disabled = false;
                
                // Validar tamaño del archivo (10MB máximo)
                if (file.size > 10 * 1024 * 1024) {
                    alert('El archivo es demasiado grande. Máximo 10MB.');
                    clearFile();
                    return;
                }
                
                // Validar extensión
                const validExtensions = ['.xlsx', '.xls', '.csv'];
                const fileExtension = '.' + file.name.split('.').pop().toLowerCase();
                if (!validExtensions.includes(fileExtension)) {
                    alert('Formato de archivo no válido. Solo se permiten archivos .xlsx, .xls o .csv');
                    clearFile();
                    return;
                }
            } else {
                clearFile();
            }
        });
        
        function clearFile() {
            fileInput.value = '';
            fileInfo.style.display = 'none';
            submitBtn.disabled = true;
        }
        
        // Drag and drop
        uploadArea.addEventListener('dragover', function(e) {
            e.preventDefault();
            this.classList.add('dragover');
        });
        
        uploadArea.addEventListener('dragleave', function(e) {
            e.preventDefault();
            this.classList.remove('dragover');
        });
        
        uploadArea.addEventListener('drop', function(e) {
            e.preventDefault();
            this.classList.remove('dragover');
            
            const files = e.dataTransfer.files;
            if (files && files.length > 0) {
                // Asignar archivos al input
                fileInput.files = files;
                
                // Disparar evento change manualmente para activar validaciones
                const changeEvent = new Event('change', { bubbles: true });
                fileInput.dispatchEvent(changeEvent);
            }
        });
        
        uploadArea.addEventListener('click', function(e) {
            // Solo activar si no se hizo clic en el botón
            if (e.target.tagName !== 'BUTTON' && !e.target.closest('button')) {
                fileInput.click();
            }
        });
        
        // Manejo del envío del formulario
        document.getElementById('uploadForm').addEventListener('submit', function(e) {
            const submitBtn = document.getElementById('submitBtn');
            const submitIcon = document.getElementById('submitIcon');
            const submitText = document.getElementById('submitText');
            
            // Prevenir envíos múltiples
            if (submitBtn.disabled) {
                e.preventDefault();
                return false;
            }
            
            // Cambiar estado del botón
            submitBtn.disabled = true;
            submitIcon.className = 'fas fa-spinner fa-spin me-2';
            submitText.textContent = 'Procesando...';
            
            // Opcional: restaurar estado después de un tiempo si no hay redirección
            setTimeout(function() {
                if (submitBtn.disabled) {
                    submitBtn.disabled = false;
                    submitIcon.className = 'fas fa-upload me-2';
                    submitText.textContent = 'Importar Contactos';
                }
            }, 30000); // 30 segundos
         });
    </script>
</body>
</html>