<?php
require_once 'config.php';

// Verificar si el usuario está logueado
if (!isLoggedIn()) {
    header('Location: login.php');
    exit();
}

// Inicializar conexión a la base de datos
$database = new Database();
$db = $database->getConnection();

// Verificar estructura de base de datos
$stmt = $db->query("SHOW COLUMNS FROM contactos LIKE 'estado'");
$has_estado = $stmt->rowCount() > 0;

$stmt = $db->query("SHOW COLUMNS FROM contactos LIKE 'observaciones'");
$has_observaciones = $stmt->rowCount() > 0;

$stmt = $db->query("SHOW TABLES LIKE 'etiquetas'");
$has_etiquetas_table = $stmt->rowCount() > 0;

$stmt = $db->query("SHOW TABLES LIKE 'historial_cambios'");
$has_historial_table = $stmt->rowCount() > 0;

// Función para registrar cambios (si la tabla existe)
function registrarCambio($db, $has_historial_table, $contacto_id, $campo, $valor_anterior, $valor_nuevo, $usuario_id) {
    if (!$has_historial_table) return;
    
    $stmt = $db->prepare("INSERT INTO historial_cambios (contacto_id, campo_modificado, valor_anterior, valor_nuevo, usuario_id, fecha_cambio) VALUES (?, ?, ?, ?, ?, NOW())");
    $stmt->execute([$contacto_id, $campo, $valor_anterior, $valor_nuevo, $usuario_id]);
}

// Procesar acciones AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    // Verificar que el usuario existe para todas las acciones AJAX
    $stmt = $db->prepare("SELECT id FROM usuarios WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    if ($stmt->rowCount() == 0) {
        echo json_encode(['success' => false, 'message' => 'Usuario no válido. Por favor, inicie sesión nuevamente.']);
        exit;
    }
    
    switch ($_POST['action']) {
        case 'obtener_historial':
            if (!$has_historial_table) {
                echo json_encode(['success' => false, 'message' => 'Historial no disponible']);
                exit;
            }
            
            $contacto_id = $_POST['contacto_id'];
            $stmt = $db->prepare("SELECT h.*, u.username as usuario_nombre FROM historial_cambios h LEFT JOIN usuarios u ON h.usuario_id = u.id WHERE h.contacto_id = ? ORDER BY h.fecha_cambio DESC LIMIT 10");
            $stmt->execute([$contacto_id]);
            $historial = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['success' => true, 'historial' => $historial]);
            exit;
            
        case 'actualizar_estado':
            if (!$has_estado) {
                echo json_encode(['success' => false, 'message' => 'Campo estado no disponible']);
                exit;
            }
            
            $contacto_id = $_POST['contacto_id'];
            $nuevo_estado = $_POST['estado'];
            
            // Obtener estado anterior
            $stmt = $db->prepare("SELECT estado FROM contactos WHERE id = ?");
            $stmt->execute([$contacto_id]);
            $contacto = $stmt->fetch(PDO::FETCH_ASSOC);
            $estado_anterior = $contacto['estado'] ?? '';
            
            // Actualizar estado
            $stmt = $db->prepare("UPDATE contactos SET estado = ? WHERE id = ?");
            $stmt->execute([$nuevo_estado, $contacto_id]);
            
            // Registrar cambio
            registrarCambio($db, $has_historial_table, $contacto_id, 'estado', $estado_anterior, $nuevo_estado, $_SESSION['user_id']);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'actualizar_observaciones':
            if (!$has_observaciones) {
                echo json_encode(['success' => false, 'message' => 'Campo observaciones no disponible']);
                exit;
            }
            
            $contacto_id = $_POST['contacto_id'];
            $nuevas_observaciones = $_POST['observaciones'];
            
            // Obtener observaciones anteriores
            $stmt = $db->prepare("SELECT observaciones FROM contactos WHERE id = ?");
            $stmt->execute([$contacto_id]);
            $contacto = $stmt->fetch(PDO::FETCH_ASSOC);
            $observaciones_anteriores = $contacto['observaciones'] ?? '';
            
            // Actualizar observaciones
            $stmt = $db->prepare("UPDATE contactos SET observaciones = ? WHERE id = ?");
            $stmt->execute([$nuevas_observaciones, $contacto_id]);
            
            // Registrar cambio
            registrarCambio($db, $has_historial_table, $contacto_id, 'observaciones', $observaciones_anteriores, $nuevas_observaciones, $_SESSION['user_id']);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'obtener_etiquetas_contacto':
            if (!$has_etiquetas_table) {
                echo json_encode(['success' => false, 'message' => 'Etiquetas no disponibles']);
                exit;
            }
            
            $contacto_id = $_POST['contacto_id'];
            
            // Obtener todas las etiquetas
            $stmt = $db->prepare("SELECT * FROM etiquetas ORDER BY nombre");
            $stmt->execute();
            $etiquetas = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Obtener etiquetas del contacto
            $stmt = $db->prepare("SELECT etiqueta_id FROM contacto_etiquetas WHERE contacto_id = ?");
            $stmt->execute([$contacto_id]);
            $contacto_etiquetas = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            echo json_encode([
                'success' => true, 
                'etiquetas' => $etiquetas,
                'contacto_etiquetas' => array_map('intval', $contacto_etiquetas)
            ]);
            exit;
            
        case 'actualizar_etiquetas_contacto':
            if (!$has_etiquetas_table) {
                echo json_encode(['success' => false, 'message' => 'Etiquetas no disponibles']);
                exit;
            }
            
            $contacto_id = $_POST['contacto_id'];
            $etiquetas_nuevas = isset($_POST['etiquetas']) ? $_POST['etiquetas'] : [];
            
            // Obtener etiquetas anteriores
            $stmt = $db->prepare("SELECT e.nombre FROM contacto_etiquetas ce JOIN etiquetas e ON ce.etiqueta_id = e.id WHERE ce.contacto_id = ?");
            $stmt->execute([$contacto_id]);
            $etiquetas_anteriores = $stmt->fetchAll(PDO::FETCH_COLUMN);
            
            // Eliminar etiquetas existentes
            $stmt = $db->prepare("DELETE FROM contacto_etiquetas WHERE contacto_id = ?");
            $stmt->execute([$contacto_id]);
            
            // Agregar nuevas etiquetas
            if (!empty($etiquetas_nuevas)) {
                $stmt = $db->prepare("INSERT INTO contacto_etiquetas (contacto_id, etiqueta_id) VALUES (?, ?)");
                foreach ($etiquetas_nuevas as $etiqueta_id) {
                    $stmt->execute([$contacto_id, $etiqueta_id]);
                }
            }
            
            // Obtener nombres de las nuevas etiquetas
            if (!empty($etiquetas_nuevas)) {
                $placeholders = str_repeat('?,', count($etiquetas_nuevas) - 1) . '?';
                $stmt = $db->prepare("SELECT nombre FROM etiquetas WHERE id IN ($placeholders)");
                $stmt->execute($etiquetas_nuevas);
                $nombres_nuevas = $stmt->fetchAll(PDO::FETCH_COLUMN);
            } else {
                $nombres_nuevas = [];
            }
            
            // Registrar cambio
            $valor_anterior = implode(', ', $etiquetas_anteriores);
            $valor_nuevo = implode(', ', $nombres_nuevas);
            registrarCambio($db, $has_historial_table, $contacto_id, 'etiquetas', $valor_anterior, $valor_nuevo, $_SESSION['user_id']);
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'obtener_contacto':
            $contacto_id = $_POST['contacto_id'];
            
            $stmt = $db->prepare("SELECT * FROM contactos WHERE id = ?");
            $stmt->execute([$contacto_id]);
            $contacto = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($contacto) {
                echo json_encode(['success' => true, 'contacto' => $contacto]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Contacto no encontrado']);
            }
            exit;
            
        case 'actualizar_contacto':
            $contacto_id = $_POST['contacto_id'];
            $nombres_apellidos = $_POST['nombres_apellidos'];
            $celular = $_POST['celular'];
            $ciudad = $_POST['ciudad'];
            $fecha_registro = $_POST['fecha_registro'];
            
            // Obtener datos anteriores para el historial
            $stmt = $db->prepare("SELECT * FROM contactos WHERE id = ?");
            $stmt->execute([$contacto_id]);
            $contacto_anterior = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Actualizar contacto
            $sql = "UPDATE contactos SET nombres_apellidos = ?, celular = ?, ciudad = ?, fecha_registro = ?";
            $params = [$nombres_apellidos, $celular, $ciudad, $fecha_registro];
            
            if ($has_estado && isset($_POST['estado'])) {
                $sql .= ", estado = ?";
                $params[] = $_POST['estado'];
            }
            
            if ($has_observaciones && isset($_POST['observaciones'])) {
                $sql .= ", observaciones = ?";
                $params[] = $_POST['observaciones'];
            }
            
            $sql .= " WHERE id = ?";
            $params[] = $contacto_id;
            
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            
            // Registrar cambios en el historial
            if ($contacto_anterior['nombres_apellidos'] != $nombres_apellidos) {
                registrarCambio($db, $has_historial_table, $contacto_id, 'nombres_apellidos', $contacto_anterior['nombres_apellidos'], $nombres_apellidos, $_SESSION['user_id']);
            }
            if ($contacto_anterior['celular'] != $celular) {
                registrarCambio($db, $has_historial_table, $contacto_id, 'celular', $contacto_anterior['celular'], $celular, $_SESSION['user_id']);
            }
            if ($contacto_anterior['ciudad'] != $ciudad) {
                registrarCambio($db, $has_historial_table, $contacto_id, 'ciudad', $contacto_anterior['ciudad'], $ciudad, $_SESSION['user_id']);
            }
            if ($contacto_anterior['fecha_registro'] != $fecha_registro) {
                registrarCambio($db, $has_historial_table, $contacto_id, 'fecha_registro', $contacto_anterior['fecha_registro'], $fecha_registro, $_SESSION['user_id']);
            }
            
            if ($has_estado && isset($_POST['estado']) && $contacto_anterior['estado'] != $_POST['estado']) {
                registrarCambio($db, $has_historial_table, $contacto_id, 'estado', $contacto_anterior['estado'], $_POST['estado'], $_SESSION['user_id']);
            }
            
            if ($has_observaciones && isset($_POST['observaciones']) && $contacto_anterior['observaciones'] != $_POST['observaciones']) {
                registrarCambio($db, $has_historial_table, $contacto_id, 'observaciones', $contacto_anterior['observaciones'], $_POST['observaciones'], $_SESSION['user_id']);
            }
            
            echo json_encode(['success' => true]);
            exit;
            
        case 'marcar_separo':
            $contacto_id = $_POST['contacto_id'];
            
            // Obtener nombre del contacto
            $stmt = $db->prepare("SELECT nombres_apellidos FROM contactos WHERE id = ?");
            $stmt->execute([$contacto_id]);
            $contacto = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($contacto) {
                // Registrar en el historial
                registrarCambio($db, $has_historial_table, $contacto_id, 'accion', '', 'Separó', $_SESSION['user_id']);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Contacto no encontrado']);
            }
            exit;
            
        case 'marcar_compro':
            $contacto_id = $_POST['contacto_id'];
            
            // Obtener nombre del contacto
            $stmt = $db->prepare("SELECT nombres_apellidos FROM contactos WHERE id = ?");
            $stmt->execute([$contacto_id]);
            $contacto = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($contacto) {
                // Registrar en el historial
                registrarCambio($db, $has_historial_table, $contacto_id, 'accion', '', 'Compró', $_SESSION['user_id']);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Contacto no encontrado']);
            }
            exit;
    }
    exit;
}// Procesar formulario de nuevo contacto
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !isset($_POST['action'])) {
    try {
        // Verificar que el usuario existe en la base de datos
        $stmt = $db->prepare("SELECT id FROM usuarios WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        if ($stmt->rowCount() == 0) {
            throw new Exception("Usuario no válido. Por favor, inicie sesión nuevamente.");
        }
        
        $nombres_apellidos = $_POST['nombres_apellidos'];
        $celular = $_POST['celular'];
        $ciudad = $_POST['ciudad'];
        $fecha_registro = $_POST['fecha_registro'];
        
        // Construir query dinámicamente según campos disponibles
        $campos = ['usuario_id', 'nombres_apellidos', 'celular', 'ciudad', 'fecha_registro'];
        $valores = [$_SESSION['user_id'], $nombres_apellidos, $celular, $ciudad, $fecha_registro];
        $placeholders = ['?', '?', '?', '?', '?'];
        
        if ($has_estado && isset($_POST['estado'])) {
            $campos[] = 'estado';
            $valores[] = $_POST['estado'];
            $placeholders[] = '?';
        }
        
        if ($has_observaciones && isset($_POST['observaciones'])) {
            $campos[] = 'observaciones';
            $valores[] = $_POST['observaciones'];
            $placeholders[] = '?';
        }
        
        $query = "INSERT INTO contactos (" . implode(', ', $campos) . ") VALUES (" . implode(', ', $placeholders) . ")";
        $stmt = $db->prepare($query);
        $stmt->execute($valores);
        
        $contacto_id = $db->lastInsertId();
        
        // Manejar etiquetas si están disponibles
        if ($has_etiquetas_table && isset($_POST['etiquetas']) && !empty($_POST['etiquetas'])) {
            $stmt = $db->prepare("INSERT INTO contacto_etiquetas (contacto_id, etiqueta_id) VALUES (?, ?)");
            foreach ($_POST['etiquetas'] as $etiqueta_id) {
                $stmt->execute([$contacto_id, $etiqueta_id]);
            }
            
            // Registrar en historial
            $stmt = $db->prepare("SELECT nombre FROM etiquetas WHERE id IN (" . str_repeat('?,', count($_POST['etiquetas']) - 1) . "?)");
            $stmt->execute($_POST['etiquetas']);
            $nombres_etiquetas = $stmt->fetchAll(PDO::FETCH_COLUMN);
            $valor_etiquetas = implode(', ', $nombres_etiquetas);
            registrarCambio($db, $has_historial_table, $contacto_id, 'etiquetas', '', $valor_etiquetas, $_SESSION['user_id']);
        }
        
        $success_message = "Contacto agregado exitosamente";
    } catch (Exception $e) {
        $error_message = "Error al agregar contacto: " . $e->getMessage();
    }
}

// Obtener contactos
try {
    // Verificar que el usuario existe antes de obtener contactos
    $stmt = $db->prepare("SELECT id FROM usuarios WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    if ($stmt->rowCount() == 0) {
        throw new Exception("Usuario no válido. Por favor, inicie sesión nuevamente.");
    }
    
    // Construir SELECT dinámicamente
    $select_campos = ['c.id', 'c.nombres_apellidos', 'c.celular', 'c.ciudad', 'c.fecha_registro'];
    
    if ($has_estado) {
        $select_campos[] = 'c.estado';
    }
    
    if ($has_observaciones) {
        $select_campos[] = 'c.observaciones';
    }
    
    $query = "SELECT " . implode(', ', $select_campos) . " FROM contactos c WHERE c.usuario_id = ? ORDER BY c.fecha_registro DESC";
    $stmt = $db->prepare($query);
    $stmt->execute([$_SESSION['user_id']]);
    $contactos = $stmt->fetchAll();
    
    // Agregar valores por defecto si no existen
    foreach ($contactos as &$contacto) {
        if (!$has_estado) {
            $contacto['estado'] = 'N/A';
        }
        if (!$has_observaciones) {
            $contacto['observaciones'] = '';
        }
        $contacto['etiquetas'] = []; // Por simplicidad, vacío si no hay tabla
    }
    
} catch (Exception $e) {
    $error_message = "Error al obtener contactos: " . $e->getMessage();
    $contactos = [];
}

// Obtener etiquetas disponibles (si la tabla existe)
$etiquetas = [];
if ($has_etiquetas_table) {
    $stmt = $db->query("SELECT * FROM etiquetas ORDER BY nombre");
    $etiquetas = $stmt->fetchAll(PDO::FETCH_ASSOC);
}
?>

<?php
require_once 'settings.php';
$systemSettings = getSystemSettings();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Contactos - CRM</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="dynamic_styles.php" rel="stylesheet" type="text/css">
    <style>
        :root {
            --primary-color: #dc2626;
            --secondary-color: #b91c1c;
            --accent-color: #f87171;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --dark-color: #1f2937;
            --light-color: #f8fafc;
            --border-color: #e5e7eb;
            --text-primary: #111827;
            --text-secondary: #6b7280;
            --header-height: 70px;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            font-family: 'Poppins', sans-serif;
        }
        
        body {
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            color: var(--text-primary);
            min-height: 100vh;
            padding-top: var(--header-height);
        }
        
        /* Estilos específicos de contactos */
        
        .main-content {
            width: 100%;
            min-height: calc(100vh - var(--header-height));
            padding: 0;
        }
        
        @media (max-width: 768px) {
            .top-header {
                padding: 0 1rem;
            }
            
            .nav-menu {
                position: absolute;
                top: 100%;
                left: 0;
                right: 0;
                background: var(--primary-color);
                flex-direction: column;
                gap: 0;
                display: none;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
            }
            
            .nav-menu.show {
                display: flex;
            }
            
            .nav-menu a {
                padding: 1rem 2rem;
                border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            }
            
            .mobile-menu-toggle {
                display: block;
            }
        }
        
        @media (max-width: 480px) {
            .logo-container h4 {
                display: none;
            }
            
            .user-dropdown span {
                display: none;
            }
        }
        
        .content-area {
            padding: 2rem;
        }
        
        @media (max-width: 768px) {
            .content-area {
                padding: 0;
            }
        }
        
        .card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: 16px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            margin-bottom: 1.5rem;
        }
        
        .card:hover {
            transform: translateY(-2px);
            box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
        }
        
        /* Mobile Cards */
        @media (max-width: 768px) {
            .card {
                margin: 0.75rem;
                border-radius: 12px;
            }
            
            .card:hover {
                transform: none;
            }
        }
        
        .card-header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border-radius: 16px 16px 0 0 !important;
            border: none;
            padding: 1.5rem;
            font-weight: 600;
        }
        
        .table {
            border-radius: 12px;
            overflow: hidden;
        }
        
        .table th {
            background: linear-gradient(135deg, #f8fafc, #e2e8f0);
            border: none;
            font-weight: 600;
            color: var(--text-primary);
            padding: 1rem;
            white-space: nowrap;
        }
        
        .table td {
            padding: 1rem;
            border-color: var(--border-color);
            vertical-align: middle;
        }
        
        /* Responsive Table */
        @media (max-width: 768px) {
            .table-responsive {
                border-radius: 12px;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
            }
            
            .table th,
            .table td {
                padding: 0.75rem 0.5rem;
                font-size: 0.875rem;
            }
            
            .table th {
                font-size: 0.8rem;
            }
            
            /* Ocultar columnas menos importantes en móvil */
            .table .d-none-mobile {
                display: none;
            }
            
            .btn-sm {
                padding: 0.375rem 0.75rem;
                font-size: 0.75rem;
            }
        }
        
        .btn {
            font-weight: 500;
            border-radius: 10px;
            padding: 0.75rem 1.5rem;
            transition: all 0.3s ease;
            border: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            box-shadow: 0 4px 15px rgba(99, 102, 241, 0.3);
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(99, 102, 241, 0.4);
        }
        
        .btn-sm {
            padding: 0.5rem 1rem;
            font-size: 0.875rem;
            border-radius: 8px;
        }
        
        .estado-frio { 
            background: linear-gradient(135deg, #dbeafe, #bfdbfe);
            color: #1e40af;
        }
        
        .estado-tibio { 
            background: linear-gradient(135deg, #fef3c7, #fde68a);
            color: #92400e;
        }
        
        .estado-caliente { 
            background: linear-gradient(135deg, #fecaca, #fca5a5);
            color: #991b1b;
        }
        
        .alert {
            border-radius: 12px;
            border: none;
            font-weight: 500;
        }
        
        .alert-info {
            background: linear-gradient(135deg, #dbeafe, #bfdbfe);
            color: #1e40af;
        }
        
        .alert-success {
            background: linear-gradient(135deg, #d1fae5, #a7f3d0);
            color: #065f46;
        }
        
        .alert-danger {
            background: linear-gradient(135deg, #fecaca, #fca5a5);
            color: #991b1b;
        }
        
        .form-control, .form-select {
            border-radius: 10px;
            border: 2px solid var(--border-color);
            padding: 0.75rem 1rem;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            font-weight: 400;
        }
        
        .form-control:focus, .form-select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.2rem rgba(99, 102, 241, 0.25);
        }
        
        /* Mobile Overlay */
        .mobile-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 999;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .mobile-overlay.active {
            display: block;
            opacity: 1;
        }
        
        @media (max-width: 768px) {
            .form-control, .form-select {
                padding: 0.875rem 1rem;
                font-size: 16px; /* Previene zoom en iOS */
            }
            
            .modal-dialog {
                margin: 1rem;
                max-width: calc(100% - 2rem);
            }
            
            .container-fluid {
                padding: 1rem;
            }
            
            /* Mejoras adicionales para móviles */
            .card-header {
                background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
                color: white;
                border-radius: 12px 12px 0 0 !important;
                padding: 1rem;
            }
            
            .table th {
                background-color: #f8f9fa;
                font-weight: 600;
                color: #495057;
                border-color: #f0f0f0;
            }
            
            .table td {
                border-color: #f0f0f0;
            }
            
            .btn-sm {
                min-width: 36px;
                height: 36px;
                border-radius: 6px;
            }
            
            .gap-1 {
                gap: 0.25rem !important;
            }
            
            .text-muted.small {
                font-size: 0.75rem;
                line-height: 1.4;
                margin-top: 0.25rem;
            }
            
            .text-muted.small i {
                width: 12px;
                margin-right: 4px;
            }
            
            /* Estados de contacto */
            .estado-frio {
                border-left: 4px solid #6c757d;
            }
            
            .estado-tibio {
                border-left: 4px solid #fd7e14;
            }
            
            .estado-caliente {
                border-left: 4px solid #dc3545;
            }
            
            /* Navbar móvil mejorada */
             .navbar {
                 backdrop-filter: blur(10px);
                 background-color: rgba(255, 255, 255, 0.95) !important;
                 box-shadow: 0 2px 10px rgba(0,0,0,0.1);
             }
             
             .toggle-btn {
                 padding: 0.5rem;
                 border-radius: 8px;
                 transition: all 0.2s ease;
             }
             
             .toggle-btn.active {
                 transform: scale(0.95);
             }
             
             /* Filtros móviles */
             .filter-btn {
                 border-radius: 20px !important;
                 font-weight: 500;
                 transition: all 0.3s ease;
             }
             
             .filter-btn.active {
                 transform: translateY(-2px);
                 box-shadow: 0 4px 12px rgba(0,0,0,0.15);
             }
             
             /* Vista de tarjetas móviles */
             .contact-card {
                 margin-bottom: 1rem;
                 border-radius: 12px;
                 box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
                 transition: all 0.3s ease;
                 border-left: 4px solid #dee2e6;
                 animation: fadeIn 0.3s ease-in;
             }
             
             .contact-card:hover {
                 transform: translateY(-2px);
                 box-shadow: 0 4px 16px rgba(0, 0, 0, 0.15);
             }
             
             .contact-card.estado-frio {
                 border-left-color: #6c757d;
                 background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
             }
             
             .contact-card.estado-tibio {
                 border-left-color: #fd7e14;
                 background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
             }
             
             .contact-card.estado-caliente {
                 border-left-color: #dc3545;
                 background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
             }
             
             .contact-header {
                 display: flex;
                 justify-content: space-between;
                 align-items: center;
                 margin-bottom: 1rem;
                 padding-bottom: 0.5rem;
                 border-bottom: 1px solid rgba(0, 0, 0, 0.1);
             }
             
             .contact-name {
                 margin: 0;
                 font-weight: 600;
                 color: #2c3e50;
                 font-size: 1.1rem;
             }
             
             .contact-status {
                 padding: 0.25rem 0.75rem;
                 border-radius: 20px;
                 font-size: 0.8rem;
                 font-weight: 500;
                 text-transform: uppercase;
                 letter-spacing: 0.5px;
             }
             
             .status-frio {
                 background: #6c757d;
                 color: white;
             }
             
             .status-tibio {
                 background: #fd7e14;
                 color: white;
             }
             
             .status-caliente {
                 background: #dc3545;
                 color: white;
             }
             
             .contact-info {
                 margin-bottom: 1rem;
             }
             
             .contact-info-item {
                 display: flex;
                 align-items: center;
                 margin-bottom: 0.5rem;
                 font-size: 0.9rem;
                 color: #495057;
             }
             
             .contact-info-item i {
                 width: 20px;
                 margin-right: 0.75rem;
                 color: #6c757d;
                 text-align: center;
             }
             
             .contact-actions {
                 display: flex;
                 gap: 0.5rem;
                 flex-wrap: wrap;
                 justify-content: flex-end;
             }
             
             .contact-actions .btn {
                 border-radius: 8px;
                 padding: 0.375rem 0.75rem;
                 font-size: 0.875rem;
                 transition: all 0.2s ease;
             }
             
             .contact-actions .btn:hover {
                 transform: translateY(-1px);
                 box-shadow: 0 2px 8px rgba(0, 0, 0, 0.15);
             }
             
             /* Animaciones */
              @keyframes fadeIn {
                  from {
                      opacity: 0;
                      transform: translateY(10px);
                  }
                  to {
                      opacity: 1;
                      transform: translateY(0);
                  }
              }
              
              /* Estilos para barra de búsqueda */
              .search-container .input-group {
                  border-radius: 12px;
                  overflow: hidden;
                  box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
              }
              
              .search-container .input-group-text {
                  border: none;
                  background: #f8f9fa;
              }
              
              .search-container .form-control {
                  border: none;
                  background: #f8f9fa;
                  font-size: 16px;
                  padding: 0.75rem;
              }
              
              .search-container .form-control:focus {
                  box-shadow: none;
                  background: white;
              }
              
              .search-container .btn {
                  border: none;
                  background: #f8f9fa;
                  transition: all 0.2s ease;
              }
              
              .search-container .btn:hover {
                  background: #e9ecef;
                  transform: scale(1.05);
              }
              
              .contact-card {
                  border: none;
                  border-radius: 12px;
                  box-shadow: 0 2px 8px rgba(0,0,0,0.1);
                  margin-bottom: 1rem;
                  transition: all 0.3s ease;
              }
             
             .contact-card:hover {
                 transform: translateY(-2px);
                 box-shadow: 0 4px 16px rgba(0,0,0,0.15);
             }
             
             .contact-card .card-body {
                 padding: 1rem;
             }
             
             .contact-header {
                 display: flex;
                 justify-content: space-between;
                 align-items: flex-start;
                 margin-bottom: 0.75rem;
             }
             
             .contact-name {
                 font-weight: 600;
                 font-size: 1.1rem;
                 color: #2d3748;
                 margin: 0;
             }
             
             .contact-status {
                 padding: 0.25rem 0.75rem;
                 border-radius: 20px;
                 font-size: 0.75rem;
                 font-weight: 600;
             }
             
             .status-frio {
                 background-color: #e2e8f0;
                 color: #4a5568;
             }
             
             .status-tibio {
                 background-color: #fed7aa;
                 color: #c2410c;
             }
             
             .status-caliente {
                 background-color: #fecaca;
                 color: #dc2626;
             }
             
             .contact-info {
                 display: flex;
                 flex-direction: column;
                 gap: 0.5rem;
                 margin-bottom: 1rem;
             }
             
             .contact-info-item {
                 display: flex;
                 align-items: center;
                 font-size: 0.875rem;
                 color: #6b7280;
             }
             
             .contact-info-item i {
                 width: 16px;
                 margin-right: 0.5rem;
                 color: var(--primary-color);
             }
             
             .contact-actions {
                 display: flex;
                 gap: 0.5rem;
                 flex-wrap: wrap;
             }
         }
        
        .modal-content {
            border-radius: 16px;
            border: none;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.2);
        }
        
        .modal-header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border-radius: 16px 16px 0 0;
            border: none;
        }
        
        .btn-close {
            filter: invert(1);
        }
        
        .settings-panel {
            position: fixed;
            top: 0;
            right: -400px;
            width: 400px;
            height: 100vh;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            box-shadow: -10px 0 30px rgba(0, 0, 0, 0.1);
            transition: right 0.3s ease;
            z-index: 1060;
            padding: 2rem;
            overflow-y: auto;
        }
        
        .settings-panel.active {
            right: 0;
        }
        
        .settings-toggle {
            position: fixed;
            top: 50%;
            right: 20px;
            transform: translateY(-50%);
            background: var(--primary-color);
            color: white;
            border: none;
            border-radius: 50%;
            width: 60px;
            height: 60px;
            font-size: 1.5rem;
            box-shadow: 0 4px 20px rgba(99, 102, 241, 0.3);
            transition: all 0.3s ease;
            z-index: 1061;
        }
        
        .settings-toggle:hover {
            transform: translateY(-50%) scale(1.1);
            box-shadow: 0 6px 25px rgba(99, 102, 241, 0.4);
        }
        
        .color-picker {
            width: 50px;
            height: 50px;
            border-radius: 10px;
            border: 2px solid var(--border-color);
            cursor: pointer;
        }
        
        .logo-preview {
            width: 100px;
            height: 100px;
            border: 2px dashed var(--border-color);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            background-size: contain;
            background-repeat: no-repeat;
            background-position: center;
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="content-area">

    <!-- Botón de configuración visual -->
    <button class="settings-toggle" onclick="toggleSettings()">
        <i class="fas fa-cog"></i>
    </button>

    <!-- Panel de configuración visual -->
    <div class="settings-panel" id="settingsPanel">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h4 class="mb-0"><i class="fas fa-palette"></i> Configuración Visual</h4>
            <button class="btn btn-sm btn-outline-secondary" onclick="toggleSettings()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <!-- Colores del tema -->
        <div class="mb-4">
            <h6 class="fw-bold mb-3">Colores del Tema</h6>
            <div class="row g-3">
                <div class="col-6">
                    <label class="form-label small">Color Primario</label>
                    <input type="color" class="color-picker" id="primaryColor" value="#6366f1" onchange="updateColor('primary', this.value)">
                </div>
                <div class="col-6">
                    <label class="form-label small">Color Secundario</label>
                    <input type="color" class="color-picker" id="secondaryColor" value="#8b5cf6" onchange="updateColor('secondary', this.value)">
                </div>
                <div class="col-6">
                    <label class="form-label small">Color de Acento</label>
                    <input type="color" class="color-picker" id="accentColor" value="#06b6d4" onchange="updateColor('accent', this.value)">
                </div>
                <div class="col-6">
                    <label class="form-label small">Color de Éxito</label>
                    <input type="color" class="color-picker" id="successColor" value="#10b981" onchange="updateColor('success', this.value)">
                </div>
            </div>
        </div>
        
        <!-- Logo personalizado -->
        <div class="mb-4">
            <h6 class="fw-bold mb-3">Logo Personalizado</h6>
            <div class="logo-preview mb-3" id="logoPreview">
                <i class="fas fa-image text-muted"></i>
            </div>
            <input type="file" class="form-control" id="logoUpload" accept="image/*" onchange="updateLogo(this)">
            <small class="text-muted">Formatos: PNG, JPG, SVG (máx. 2MB)</small>
        </div>
        
        <!-- Nombre de la empresa -->
        <div class="mb-4">
            <h6 class="fw-bold mb-3">Información de la Empresa</h6>
            <div class="mb-3">
                <label class="form-label small">Nombre de la Empresa</label>
                <input type="text" class="form-control" id="companyName" value="Mi Empresa" onchange="updateCompanyName(this.value)">
            </div>
        </div>
        
        <!-- Tipografía -->
        <div class="mb-4">
            <h6 class="fw-bold mb-3">Tipografía</h6>
            <select class="form-select" id="fontFamily" onchange="updateFont(this.value)">
                <option value="Poppins" selected>Poppins (Actual)</option>
                <option value="Inter">Inter</option>
                <option value="Roboto">Roboto</option>
                <option value="Open Sans">Open Sans</option>
                <option value="Montserrat">Montserrat</option>
            </select>
        </div>
        
        <!-- Acciones -->
        <div class="d-grid gap-2">
            <button class="btn btn-primary" onclick="saveSettings()">
                <i class="fas fa-save"></i> Guardar Configuración
            </button>
            <button class="btn btn-outline-secondary" onclick="resetSettings()">
                <i class="fas fa-undo"></i> Restaurar por Defecto
            </button>
        </div>
    </div>

        <div class="container-fluid p-4">
        <!-- Alertas -->
        <?php if (isset($success_message)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle"></i> <?php echo $success_message; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-circle"></i> <?php echo $error_message; ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Información del sistema -->
        <?php if (!$has_estado || !$has_observaciones || !$has_etiquetas_table): ?>
        <div class="alert alert-info" role="alert">
            <i class="fas fa-info-circle"></i> <strong>Información:</strong> 
            Algunas funciones están limitadas por la estructura de la base de datos.
            <?php if (!$has_estado): ?>Campo 'Estado' no disponible. <?php endif; ?>
            <?php if (!$has_observaciones): ?>Campo 'Observaciones' no disponible. <?php endif; ?>
            <?php if (!$has_etiquetas_table): ?>Sistema de etiquetas no disponible. <?php endif; ?>
            <a href="test_connection.php" class="alert-link">Ejecutar diagnóstico</a>
        </div>
        <?php endif; ?>

        <!-- Encabezado -->
        <div class="row mb-4">
            <div class="col">
                <h2><i class="fas fa-address-book"></i> Gestión de Contactos</h2>
                <p class="text-muted">Administra tu base de datos de contactos</p>
            </div>
            <div class="col-auto">
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#nuevoContactoModal">
                    <i class="fas fa-plus"></i> Nuevo Contacto
                </button>
            </div>
        </div>

        <!-- Barra de búsqueda mejorada -->
        <div class="card mb-3">
            <div class="card-body">
                <div class="search-container">
                    <div class="input-group">
                        <span class="input-group-text bg-light border-end-0">
                            <i class="fas fa-search text-muted"></i>
                        </span>
                        <input type="text" class="form-control border-start-0 ps-0" 
                               id="searchInput" placeholder="Buscar por nombre, teléfono o ciudad...">
                        <button class="btn btn-outline-secondary" type="button" id="clearSearch" style="display: none;">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <div class="mt-2">
                        <small class="text-muted">
                            <i class="fas fa-info-circle"></i> 
                            Resultados: <span id="searchResults"><?php echo count($contactos); ?></span> contactos
                        </small>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Filtros rápidos por estado -->
        <?php if ($has_estado): ?>
        <div class="card mb-3">
            <div class="card-body">
                <div class="d-flex flex-wrap gap-2 align-items-center">
                    <span class="fw-bold text-muted me-2">Filtrar por estado:</span>
                    <button class="btn btn-outline-secondary btn-sm filter-btn active" data-filter="todos">
                        <i class="fas fa-list"></i> Todos
                        <span class="badge bg-secondary ms-1" id="count-todos"><?php echo count($contactos); ?></span>
                    </button>
                    <button class="btn btn-outline-primary btn-sm filter-btn" data-filter="frio">
                        🧊 Frío
                        <span class="badge bg-primary ms-1" id="count-frio">
                            <?php echo count(array_filter($contactos, function($c) { return strtolower($c['estado']) == 'frio' || $c['estado'] == 'Frio'; })); ?>
                        </span>
                    </button>
                    <button class="btn btn-outline-warning btn-sm filter-btn" data-filter="tibio">
                        🌡️ Tibio
                        <span class="badge bg-warning ms-1" id="count-tibio">
                            <?php echo count(array_filter($contactos, function($c) { return strtolower($c['estado']) == 'tibio' || $c['estado'] == 'Tibio'; })); ?>
                        </span>
                    </button>
                    <button class="btn btn-outline-danger btn-sm filter-btn" data-filter="caliente">
                        🔥 Caliente
                        <span class="badge bg-danger ms-1" id="count-caliente">
                            <?php echo count(array_filter($contactos, function($c) { return strtolower($c['estado']) == 'caliente' || $c['estado'] == 'Caliente'; })); ?>
                        </span>
                    </button>
                </div>
            </div>
        </div>
        <?php endif; ?>
        
        <!-- Filtro por fechas -->
        <div class="card mb-3">
            <div class="card-body">
                <div class="row align-items-end">
                    <div class="col-md-3">
                        <label class="form-label fw-bold text-muted">Filtrar por fecha:</label>
                        <input type="date" class="form-control" id="fechaDesde" placeholder="Fecha desde">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label fw-bold text-muted">Hasta:</label>
                        <input type="date" class="form-control" id="fechaHasta" placeholder="Fecha hasta">
                    </div>
                    <div class="col-md-3">
                        <div class="d-flex gap-2 mt-2">
                            <button class="btn btn-primary btn-sm" id="aplicarFiltroFecha">
                                <i class="fas fa-filter"></i> Aplicar
                            </button>
                            <button class="btn btn-outline-secondary btn-sm" id="limpiarFiltroFecha">
                                <i class="fas fa-times"></i> Limpiar
                            </button>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="mt-2">
                            <small class="text-muted">
                                <i class="fas fa-calendar"></i> 
                                Filtrados: <span id="fechaResults"><?php echo count($contactos); ?></span> contactos
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Tabla de contactos -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-list"></i> Lista de Contactos</h5>
            </div>
            <div class="card-body">
                <?php if (count($contactos) > 0): ?>
                <!-- Vista de tarjetas para móviles -->
                <div class="d-block d-md-none" id="contactCards">
                    <?php foreach ($contactos as $contacto): ?>
                    <div class="contact-card card estado-<?php echo strtolower($contacto['estado']); ?>" data-estado="<?php echo strtolower($contacto['estado']); ?>" data-fecha="<?php echo $contacto['fecha_registro']; ?>">
                        <div class="card-body">
                            <div class="contact-header">
                                <h6 class="contact-name"><?php echo htmlspecialchars($contacto['nombres_apellidos']); ?></h6>
                                <?php if ($has_estado): ?>
                                <span class="contact-status status-<?php echo strtolower($contacto['estado']); ?>">
                                    <?php 
                                    $estado_icons = ['frio' => '🧊', 'tibio' => '🌡️', 'caliente' => '🔥'];
                                    echo $estado_icons[strtolower($contacto['estado'])] ?? ''; 
                                    ?> <?php echo ucfirst($contacto['estado']); ?>
                                </span>
                                <?php endif; ?>
                            </div>
                            
                            <div class="contact-info">
                                <div class="contact-info-item">
                                    <i class="fas fa-phone"></i>
                                    <span><?php echo htmlspecialchars($contacto['celular']); ?></span>
                                </div>
                                <div class="contact-info-item">
                                    <i class="fas fa-map-marker-alt"></i>
                                    <span><?php echo htmlspecialchars($contacto['ciudad']); ?></span>
                                </div>
                                <div class="contact-info-item">
                                    <i class="fas fa-calendar"></i>
                                    <span><?php echo date('d/m/Y', strtotime($contacto['fecha_registro'])); ?></span>
                                </div>
                                <?php if ($has_observaciones && !empty($contacto['observaciones'])): ?>
                                <div class="contact-info-item">
                                    <i class="fas fa-sticky-note"></i>
                                    <span><?php echo htmlspecialchars($contacto['observaciones']); ?></span>
                                </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="contact-actions">
                                <button class="btn btn-sm btn-outline-primary editar-btn" 
                                        data-contacto-id="<?php echo $contacto['id']; ?>"
                                        data-bs-toggle="modal" data-bs-target="#editarContactoModal"
                                        title="Editar">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-success separo-btn" 
                                        data-contacto-id="<?php echo $contacto['id']; ?>"
                                        title="Separó">
                                    <i class="fas fa-handshake"></i>
                                </button>
                                <button class="btn btn-sm btn-outline-warning compro-btn" 
                                        data-contacto-id="<?php echo $contacto['id']; ?>"
                                        title="Compró">
                                    <i class="fas fa-shopping-cart"></i>
                                </button>
                                <?php if ($has_historial_table): ?>
                                <button class="btn btn-sm btn-outline-info historial-btn" 
                                        data-contacto-id="<?php echo $contacto['id']; ?>"
                                        title="Historial">
                                    <i class="fas fa-history"></i>
                                </button>
                                <?php endif; ?>
                                <?php if ($has_etiquetas_table): ?>
                                <button class="btn btn-sm btn-outline-primary" 
                                        data-bs-toggle="modal" data-bs-target="#etiquetasModal" 
                                        data-contacto-id="<?php echo $contacto['id']; ?>"
                                        title="Etiquetas">
                                    <i class="fas fa-tags"></i>
                                </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                
                <!-- Vista de tabla para desktop -->
                <div class="table-responsive d-none d-md-block">
                    <table class="table table-hover" id="contactTable">
                        <thead>
                            <tr>
                                <th>Nombre</th>
                                <th>Celular</th>
                                <th class="d-none-mobile">Ciudad</th>
                                <th class="d-none-mobile">Fecha Registro</th>
                                <?php if ($has_estado): ?><th>Estado</th><?php endif; ?>
                                <?php if ($has_etiquetas_table): ?><th class="d-none-mobile">Etiquetas</th><?php endif; ?>
                                <?php if ($has_observaciones): ?><th class="d-none-mobile">Observaciones</th><?php endif; ?>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($contactos as $contacto): ?>
                            <tr class="<?php echo $has_estado ? 'estado-' . strtolower($contacto['estado']) : ''; ?>" data-estado="<?php echo strtolower($contacto['estado']); ?>" data-fecha="<?php echo $contacto['fecha_registro']; ?>">
                                <td>
                                    <strong><?php echo htmlspecialchars($contacto['nombres_apellidos']); ?></strong>
                                    <div class="d-block d-md-none text-muted small">
                                        <i class="fas fa-phone"></i> <?php echo htmlspecialchars($contacto['celular']); ?><br>
                                        <i class="fas fa-map-marker-alt"></i> <?php echo htmlspecialchars($contacto['ciudad']); ?><br>
                                        <i class="fas fa-calendar"></i> <?php echo date('d/m/Y', strtotime($contacto['fecha_registro'])); ?>
                                    </div>
                                </td>
                                <td class="d-none d-md-table-cell"><?php echo htmlspecialchars($contacto['celular']); ?></td>
                                <td class="d-none-mobile"><?php echo htmlspecialchars($contacto['ciudad']); ?></td>
                                <td class="d-none-mobile"><?php echo date('d/m/Y', strtotime($contacto['fecha_registro'])); ?></td>
                                
                                <?php if ($has_estado): ?>
                                <td>
                                    <select class="form-select form-select-sm estado-select" data-contacto-id="<?php echo $contacto['id']; ?>">
                                        <option value="Frio" <?php echo $contacto['estado'] == 'Frio' ? 'selected' : ''; ?>>🧊 Frío</option>
                                        <option value="Tibio" <?php echo $contacto['estado'] == 'Tibio' ? 'selected' : ''; ?>>🌡️ Tibio</option>
                                        <option value="Caliente" <?php echo $contacto['estado'] == 'Caliente' ? 'selected' : ''; ?>>🔥 Caliente</option>
                                    </select>
                                </td>
                                <?php endif; ?>
                                
                                <?php if ($has_etiquetas_table): ?>
                                <td class="d-none-mobile">
                                    <button type="button" class="btn btn-outline-primary btn-sm" data-bs-toggle="modal" data-bs-target="#etiquetasModal" data-contacto-id="<?php echo $contacto['id']; ?>">
                                        <i class="fas fa-tags"></i> Etiquetas
                                    </button>
                                </td>
                                <?php endif; ?>
                                
                                <?php if ($has_observaciones): ?>
                                <td class="d-none-mobile">
                                    <input type="text" class="form-control form-control-sm observaciones-input" 
                                           data-contacto-id="<?php echo $contacto['id']; ?>" 
                                           value="<?php echo htmlspecialchars($contacto['observaciones']); ?>"
                                           placeholder="Agregar observación...">
                                </td>
                                <?php endif; ?>
                                
                                <td>
                                    <div class="btn-group d-none d-md-flex" role="group">
                                        <button class="btn btn-sm btn-outline-primary editar-btn" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>"
                                                data-bs-toggle="modal" data-bs-target="#editarContactoModal">
                                            <i class="fas fa-edit"></i> Editar
                                        </button>
                                        <button class="btn btn-sm btn-outline-success separo-btn" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>">
                                            <i class="fas fa-handshake"></i> Separó
                                        </button>
                                        <button class="btn btn-sm btn-outline-warning compro-btn" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>">
                                            <i class="fas fa-shopping-cart"></i> Compró
                                        </button>
                                        <?php if ($has_historial_table): ?>
                                        <button class="btn btn-sm btn-outline-info historial-btn" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>">
                                            <i class="fas fa-history"></i>
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                    <!-- Botones móviles -->
                                    <div class="d-flex d-md-none flex-wrap gap-1">
                                        <button class="btn btn-sm btn-outline-primary editar-btn" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>"
                                                data-bs-toggle="modal" data-bs-target="#editarContactoModal"
                                                title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button class="btn btn-sm btn-outline-success separo-btn" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>"
                                                title="Separó">
                                            <i class="fas fa-handshake"></i>
                                        </button>
                                        <button class="btn btn-sm btn-outline-warning compro-btn" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>"
                                                title="Compró">
                                            <i class="fas fa-shopping-cart"></i>
                                        </button>
                                        <?php if ($has_historial_table): ?>
                                        <button class="btn btn-sm btn-outline-info historial-btn" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>"
                                                title="Historial">
                                            <i class="fas fa-history"></i>
                                        </button>
                                        <?php endif; ?>
                                        <?php if ($has_etiquetas_table): ?>
                                        <button class="btn btn-sm btn-outline-primary" 
                                                data-bs-toggle="modal" data-bs-target="#etiquetasModal" 
                                                data-contacto-id="<?php echo $contacto['id']; ?>"
                                                title="Etiquetas">
                                            <i class="fas fa-tags"></i>
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php else: ?>
                <div class="text-center py-4">
                    <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                    <h5 class="text-muted">No hay contactos registrados</h5>
                    <p class="text-muted">Comienza agregando tu primer contacto</p>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Modal Nuevo Contacto -->
    <div class="modal fade" id="nuevoContactoModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form method="POST">
                    <div class="modal-header">
                        <h5 class="modal-title"><i class="fas fa-user-plus"></i> Nuevo Contacto</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Nombres y Apellidos *</label>
                            <input type="text" class="form-control" name="nombres_apellidos" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Celular *</label>
                            <input type="text" class="form-control" name="celular" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Ciudad *</label>
                            <input type="text" class="form-control" name="ciudad" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Fecha de Registro *</label>
                            <input type="date" class="form-control" name="fecha_registro" id="fecha_registro" required>
                        </div>
                        
                        <?php if ($has_estado): ?>
                        <div class="mb-3">
                            <label class="form-label">Estado</label>
                            <select class="form-select" name="estado">
                                <option value="Frio">🧊 Frío</option>
                                <option value="Tibio">🌡️ Tibio</option>
                                <option value="Caliente">🔥 Caliente</option>
                            </select>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($has_etiquetas_table): ?>
                        <div class="mb-3">
                            <label class="form-label">Etiquetas</label>
                            <div class="etiquetas-container">
                                <?php foreach ($etiquetas as $etiqueta): ?>
                                <div class="form-check form-check-inline">
                                    <input class="form-check-input" type="checkbox" name="etiquetas[]" value="<?php echo $etiqueta['id']; ?>" id="etiqueta_<?php echo $etiqueta['id']; ?>">
                                    <label class="form-check-label" for="etiqueta_<?php echo $etiqueta['id']; ?>">
                                        <span class="badge" style="background-color: <?php echo $etiqueta['color']; ?>">
                                            <?php echo htmlspecialchars($etiqueta['nombre']); ?>
                                        </span>
                                    </label>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($has_observaciones): ?>
                        <div class="mb-3">
                            <label class="form-label">Observaciones</label>
                            <textarea class="form-control" name="observaciones" rows="3"></textarea>
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Guardar Contacto</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Historial -->
    <?php if ($has_historial_table): ?>
    <div class="modal fade" id="historialModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="fas fa-history"></i> Historial de Cambios</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="historialContent">Cargando...</div>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Modal Editar Contacto -->
    <div class="modal fade" id="editarContactoModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <form id="editarContactoForm">
                    <div class="modal-header">
                        <h5 class="modal-title"><i class="fas fa-user-edit"></i> Editar Contacto</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <input type="hidden" id="edit_contacto_id" name="contacto_id">
                        <div class="mb-3">
                            <label class="form-label">Nombres y Apellidos *</label>
                            <input type="text" class="form-control" id="edit_nombres_apellidos" name="nombres_apellidos" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Celular *</label>
                            <input type="text" class="form-control" id="edit_celular" name="celular" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Ciudad *</label>
                            <input type="text" class="form-control" id="edit_ciudad" name="ciudad" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Fecha de Registro *</label>
                            <input type="date" class="form-control" id="edit_fecha_registro" name="fecha_registro" required>
                        </div>
                        
                        <?php if ($has_estado): ?>
                        <div class="mb-3">
                            <label class="form-label">Estado</label>
                            <select class="form-select" id="edit_estado" name="estado">
                                <option value="Frio">🧊 Frío</option>
                                <option value="Tibio">🌡️ Tibio</option>
                                <option value="Caliente">🔥 Caliente</option>
                            </select>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($has_observaciones): ?>
                        <div class="mb-3">
                            <label class="form-label">Observaciones</label>
                            <textarea class="form-control" id="edit_observaciones" name="observaciones" rows="3"></textarea>
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Actualizar Contacto</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <script>
        // Close mobile menu on resize (específico para contactos)
        window.addEventListener('resize', function() {
            const navMenu = document.querySelector('.nav-menu');
            if (window.innerWidth > 768 && navMenu) {
                navMenu.classList.remove('show');
            }
        });
        
        // Función para mostrar el modal de perfil (placeholder)
        function showProfile() {
            alert('Función de perfil no implementada aún');
        }
        
        // Detectar cambios de tamaño de pantalla
        function handleResize() {
            const wasMobile = isMobile;
            isMobile = window.innerWidth <= 768;
            
            if (wasMobile !== isMobile) {
                const sidebar = document.getElementById('sidebar');
                const mainContent = document.getElementById('mainContent');
                const overlay = document.getElementById('mobileOverlay');
                
                if (isMobile) {
                    // Cambió a móvil
                    sidebar.classList.remove('collapsed');
                    mainContent.classList.remove('expanded');
                    closeMobileSidebar();
                } else {
                    // Cambió a desktop
                    closeMobileSidebar();
                    
                    // Restaurar estado del sidebar desde localStorage
                    const isCollapsed = localStorage.getItem('sidebarCollapsed') === 'true';
                    if (isCollapsed) {
                        sidebar.classList.add('collapsed');
                        mainContent.classList.add('expanded');
                    }
                }
            }
        }
        
         // Función para filtrar contactos por estado
         function filtrarPorEstado(estado) {
             // Filtrar tabla de escritorio
             const filas = document.querySelectorAll('#contactTable tbody tr');
             filas.forEach(fila => {
                 if (estado === 'todos') {
                     fila.style.display = '';
                 } else {
                     const estadoContacto = fila.getAttribute('data-estado');
                     if (estadoContacto === estado) {
                         fila.style.display = '';
                     } else {
                         fila.style.display = 'none';
                     }
                 }
             });
             
             // Filtrar tarjetas móviles
             const tarjetas = document.querySelectorAll('#contactCards .contact-card');
             tarjetas.forEach(tarjeta => {
                 if (estado === 'todos') {
                     tarjeta.style.display = '';
                     tarjeta.style.animation = 'fadeIn 0.3s ease-in';
                 } else {
                     const estadoContacto = tarjeta.getAttribute('data-estado');
                     if (estadoContacto === estado) {
                         tarjeta.style.display = '';
                         tarjeta.style.animation = 'fadeIn 0.3s ease-in';
                     } else {
                         tarjeta.style.display = 'none';
                     }
                 }
             });
             
             // Actualizar botones activos
             document.querySelectorAll('.filter-btn').forEach(btn => {
                 btn.classList.remove('active');
             });
             document.querySelector(`[data-filter="${estado}"]`).classList.add('active');
         }
         
         // Event listeners para los botones de filtro
          document.querySelectorAll('.filter-btn').forEach(btn => {
              btn.addEventListener('click', function() {
                  const estado = this.getAttribute('data-filter');
                  filtrarPorEstado(estado);
              });
          });
          
          // Función para filtrar contactos por fecha
          function filtrarPorFecha(fechaDesde, fechaHasta) {
              let contactosVisibles = 0;
              
              // Filtrar tabla de escritorio
              const filas = document.querySelectorAll('#contactTable tbody tr');
              filas.forEach(fila => {
                  const fechaContacto = fila.getAttribute('data-fecha');
                  let mostrar = true;
                  
                  if (fechaDesde && fechaContacto < fechaDesde) {
                      mostrar = false;
                  }
                  if (fechaHasta && fechaContacto > fechaHasta) {
                      mostrar = false;
                  }
                  
                  if (mostrar) {
                      fila.style.display = '';
                      contactosVisibles++;
                  } else {
                      fila.style.display = 'none';
                  }
              });
              
              // Filtrar tarjetas móviles
              const tarjetas = document.querySelectorAll('#contactCards .contact-card');
              tarjetas.forEach(tarjeta => {
                  const fechaContacto = tarjeta.getAttribute('data-fecha');
                  let mostrar = true;
                  
                  if (fechaDesde && fechaContacto < fechaDesde) {
                      mostrar = false;
                  }
                  if (fechaHasta && fechaContacto > fechaHasta) {
                      mostrar = false;
                  }
                  
                  if (mostrar) {
                      tarjeta.style.display = '';
                      tarjeta.style.animation = 'fadeIn 0.3s ease-in';
                  } else {
                      tarjeta.style.display = 'none';
                  }
              });
              
              // Actualizar contador
              document.getElementById('fechaResults').textContent = contactosVisibles;
          }
          
          // Event listeners para el filtro por fechas
          document.getElementById('aplicarFiltroFecha').addEventListener('click', function() {
              const fechaDesde = document.getElementById('fechaDesde').value;
              const fechaHasta = document.getElementById('fechaHasta').value;
              
              if (!fechaDesde && !fechaHasta) {
                  alert('Por favor selecciona al menos una fecha');
                  return;
              }
              
              filtrarPorFecha(fechaDesde, fechaHasta);
          });
          
          document.getElementById('limpiarFiltroFecha').addEventListener('click', function() {
              document.getElementById('fechaDesde').value = '';
              document.getElementById('fechaHasta').value = '';
              filtrarPorFecha('', '');
          });
          
          // Funcionalidad de búsqueda
          const searchInput = document.getElementById('searchInput');
          const clearSearchBtn = document.getElementById('clearSearch');
          const searchResults = document.getElementById('searchResults');
          
          function realizarBusqueda() {
              const termino = searchInput.value.toLowerCase().trim();
              const filas = document.querySelectorAll('#contactTable tbody tr');
              const tarjetas = document.querySelectorAll('#contactCards .contact-card');
              let resultados = 0;
              
              // Buscar en tabla de escritorio
              filas.forEach(fila => {
                  const nombre = fila.querySelector('td:first-child')?.textContent.toLowerCase() || '';
                  const telefono = fila.querySelector('td:nth-child(2)')?.textContent.toLowerCase() || '';
                  const ciudad = fila.querySelector('td:nth-child(3)')?.textContent.toLowerCase() || '';
                  
                  const coincide = nombre.includes(termino) || telefono.includes(termino) || ciudad.includes(termino);
                  
                  if (termino === '' || coincide) {
                      fila.style.display = '';
                      resultados++;
                  } else {
                      fila.style.display = 'none';
                  }
              });
              
              // Buscar en tarjetas móviles
              tarjetas.forEach(tarjeta => {
                  const nombre = tarjeta.querySelector('.contact-name')?.textContent.toLowerCase() || '';
                  const telefono = tarjeta.querySelector('.fa-phone')?.parentElement?.textContent.toLowerCase() || '';
                  const ciudad = tarjeta.querySelector('.fa-map-marker-alt')?.parentElement?.textContent.toLowerCase() || '';
                  
                  const coincide = nombre.includes(termino) || telefono.includes(termino) || ciudad.includes(termino);
                  
                  if (termino === '' || coincide) {
                      tarjeta.style.display = '';
                      if (termino !== '') {
                          tarjeta.style.animation = 'fadeIn 0.3s ease-in';
                      }
                  } else {
                      tarjeta.style.display = 'none';
                  }
              });
              
              // Actualizar contador de resultados
              if (termino === '') {
                  resultados = <?php echo isset($contactos) ? count($contactos) : 0; ?>;
              }
              searchResults.textContent = resultados;
              
              // Mostrar/ocultar botón de limpiar
              clearSearchBtn.style.display = termino ? 'block' : 'none';
          }
          
          // Event listeners para búsqueda
          searchInput.addEventListener('input', realizarBusqueda);
          
          clearSearchBtn.addEventListener('click', function() {
              searchInput.value = '';
              realizarBusqueda();
              searchInput.focus();
          });
        
        $(document).ready(function() {
            // Establecer fecha actual por defecto
            document.getElementById('fecha_registro').value = new Date().toISOString().split('T')[0];
            
            <?php if ($has_estado): ?>
            // Cambio de estado
            $('.estado-select').change(function() {
                const contactoId = $(this).data('contacto-id');
                const nuevoEstado = $(this).val();
                
                $.ajax({
                    url: 'contactos.php',
                    method: 'POST',
                    data: {
                        action: 'actualizar_estado',
                        contacto_id: contactoId,
                        estado: nuevoEstado
                    },
                    success: function(response) {
                        if (response.success) {
                            // Actualizar clase y atributo data-estado de la fila de tabla
                            const fila = $('[data-contacto-id="' + contactoId + '"]').closest('tr');
                            fila.removeClass('estado-frio estado-tibio estado-caliente');
                            fila.addClass('estado-' + nuevoEstado.toLowerCase());
                            fila.attr('data-estado', nuevoEstado.toLowerCase());
                            
                            // Actualizar clase y atributo data-estado de la tarjeta móvil
                            const tarjeta = $('.contact-card').filter(function() {
                                return $(this).find('[data-contacto-id="' + contactoId + '"]').length > 0;
                            });
                            tarjeta.removeClass('estado-frio estado-tibio estado-caliente');
                            tarjeta.addClass('estado-' + nuevoEstado.toLowerCase());
                            tarjeta.attr('data-estado', nuevoEstado.toLowerCase());
                            
                            // Actualizar el ícono y texto del estado en la tarjeta
                            const estadoIcons = {'frio': '🧊', 'tibio': '🌡️', 'caliente': '🔥'};
                            const estadoTexto = nuevoEstado.charAt(0).toUpperCase() + nuevoEstado.slice(1).toLowerCase();
                            tarjeta.find('.contact-status').html(estadoIcons[nuevoEstado.toLowerCase()] + ' ' + estadoTexto);
                            tarjeta.find('.contact-status').removeClass('status-frio status-tibio status-caliente');
                            tarjeta.find('.contact-status').addClass('status-' + nuevoEstado.toLowerCase());
                        } else {
                            alert('Error: ' + response.message);
                        }
                    },
                    error: function() {
                        alert('Error de conexión');
                    }
                });
        
        // Funciones del panel de configuración visual
        function toggleSettings() {
            const panel = document.getElementById('settingsPanel');
            panel.classList.toggle('active');
        }
        
        function updateColor(type, color) {
            const root = document.documentElement;
            switch(type) {
                case 'primary':
                    root.style.setProperty('--primary-color', color);
                    break;
                case 'secondary':
                    root.style.setProperty('--secondary-color', color);
                    break;
                case 'accent':
                    root.style.setProperty('--accent-color', color);
                    break;
                case 'success':
                    root.style.setProperty('--success-color', color);
                    break;
            }
            
            // Actualizar gradientes del navbar y botones
            updateGradients();
        }
        
        function updateGradients() {
            const primaryColor = getComputedStyle(document.documentElement).getPropertyValue('--primary-color').trim();
            const secondaryColor = getComputedStyle(document.documentElement).getPropertyValue('--secondary-color').trim();
            
            // Actualizar navbar
            const navbar = document.querySelector('.navbar');
            if (navbar) {
                navbar.style.background = `linear-gradient(135deg, ${primaryColor} 0%, ${secondaryColor} 100%)`;
            }
            
            // Actualizar botones primarios
            const primaryBtns = document.querySelectorAll('.btn-primary');
            primaryBtns.forEach(btn => {
                btn.style.background = `linear-gradient(135deg, ${primaryColor}, ${secondaryColor})`;
            });
            
            // Actualizar headers de cards
            const cardHeaders = document.querySelectorAll('.card-header');
            cardHeaders.forEach(header => {
                header.style.background = `linear-gradient(135deg, ${primaryColor}, ${secondaryColor})`;
            });
            
            // Actualizar headers de modales
            const modalHeaders = document.querySelectorAll('.modal-header');
            modalHeaders.forEach(header => {
                header.style.background = `linear-gradient(135deg, ${primaryColor}, ${secondaryColor})`;
            });
        }
        
        function updateLogo(input) {
            const file = input.files[0];
            if (file) {
                if (file.size > 2 * 1024 * 1024) {
                    alert('El archivo es demasiado grande. Máximo 2MB.');
                    return;
                }
                
                const reader = new FileReader();
                reader.onload = function(e) {
                    const logoUrl = e.target.result;
                    const logoPreview = document.getElementById('logoPreview');
                    logoPreview.style.backgroundImage = `url(${logoUrl})`;
                    logoPreview.innerHTML = '';
                    
                    // Actualizar logo en navbar
                    document.documentElement.style.setProperty('--logo-url', `url(${logoUrl})`);
                    
                    // Guardar en localStorage
                    localStorage.setItem('crm-logo', logoUrl);
                };
                reader.readAsDataURL(file);
            }
        }
        
        function updateCompanyName(name) {
            const brandElement = document.querySelector('.navbar-brand');
            if (brandElement) {
                // Mantener el icono y actualizar solo el texto
                const textNode = brandElement.childNodes[brandElement.childNodes.length - 1];
                if (textNode.nodeType === Node.TEXT_NODE) {
                    textNode.textContent = name;
                } else {
                    brandElement.appendChild(document.createTextNode(name));
                }
            }
            localStorage.setItem('crm-company-name', name);
        }
        
        function updateFont(fontFamily) {
            // Cargar la fuente si no es Poppins
            if (fontFamily !== 'Poppins') {
                const link = document.createElement('link');
                link.href = `https://fonts.googleapis.com/css2?family=${fontFamily.replace(' ', '+')}:wght@300;400;500;600;700&display=swap`;
                link.rel = 'stylesheet';
                document.head.appendChild(link);
            }
            
            // Aplicar la fuente
            document.documentElement.style.setProperty('--font-family', fontFamily);
            document.body.style.fontFamily = `'${fontFamily}', sans-serif`;
            
            localStorage.setItem('crm-font-family', fontFamily);
        }
        
        function saveSettings() {
            const settings = {
                primaryColor: document.getElementById('primaryColor').value,
                secondaryColor: document.getElementById('secondaryColor').value,
                accentColor: document.getElementById('accentColor').value,
                successColor: document.getElementById('successColor').value,
                companyName: document.getElementById('companyName').value,
                fontFamily: document.getElementById('fontFamily').value,
                logo: localStorage.getItem('crm-logo') || ''
            };
            
            localStorage.setItem('crm-settings', JSON.stringify(settings));
            
            // Mostrar confirmación
            const alert = document.createElement('div');
            alert.className = 'alert alert-success alert-dismissible fade show position-fixed';
            alert.style.top = '20px';
            alert.style.right = '20px';
            alert.style.zIndex = '9999';
            alert.innerHTML = `
                <i class="fas fa-check-circle"></i> Configuración guardada exitosamente
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            document.body.appendChild(alert);
            
            setTimeout(() => {
                if (alert.parentNode) {
                    alert.parentNode.removeChild(alert);
                }
            }, 3000);
        }
        
        function resetSettings() {
            if (confirm('¿Estás seguro de que quieres restaurar la configuración por defecto?')) {
                localStorage.removeItem('crm-settings');
                localStorage.removeItem('crm-logo');
                localStorage.removeItem('crm-company-name');
                localStorage.removeItem('crm-font-family');
                location.reload();
            }
        }
        
        function loadSettings() {
            const savedSettings = localStorage.getItem('crm-settings');
            if (savedSettings) {
                const settings = JSON.parse(savedSettings);
                
                // Aplicar colores
                if (settings.primaryColor) {
                    document.getElementById('primaryColor').value = settings.primaryColor;
                    updateColor('primary', settings.primaryColor);
                }
                if (settings.secondaryColor) {
                    document.getElementById('secondaryColor').value = settings.secondaryColor;
                    updateColor('secondary', settings.secondaryColor);
                }
                if (settings.accentColor) {
                    document.getElementById('accentColor').value = settings.accentColor;
                    updateColor('accent', settings.accentColor);
                }
                if (settings.successColor) {
                    document.getElementById('successColor').value = settings.successColor;
                    updateColor('success', settings.successColor);
                }
                
                // Aplicar nombre de empresa
                if (settings.companyName) {
                    document.getElementById('companyName').value = settings.companyName;
                    updateCompanyName(settings.companyName);
                }
                
                // Aplicar fuente
                if (settings.fontFamily) {
                    document.getElementById('fontFamily').value = settings.fontFamily;
                    updateFont(settings.fontFamily);
                }
                
                // Aplicar logo
                if (settings.logo) {
                    const logoPreview = document.getElementById('logoPreview');
                    logoPreview.style.backgroundImage = `url(${settings.logo})`;
                    logoPreview.innerHTML = '';
                    document.documentElement.style.setProperty('--logo-url', `url(${settings.logo})`);
                }
            }
        }
        
        // Cargar configuración al iniciar
        loadSettings();
        
        // Cerrar panel al hacer clic fuera
        document.addEventListener('click', function(e) {
            const panel = document.getElementById('settingsPanel');
            const toggle = document.querySelector('.settings-toggle');
            
            if (!panel.contains(e.target) && !toggle.contains(e.target) && panel.classList.contains('active')) {
                panel.classList.remove('active');
            }
        });
            });
            <?php endif; ?>
            
            <?php if ($has_observaciones): ?>
            // Cambio de observaciones (con delay)
            let observacionesTimeout;
            $('.observaciones-input').on('input', function() {
                const input = $(this);
                const contactoId = input.data('contacto-id');
                const nuevasObservaciones = input.val();
                
                clearTimeout(observacionesTimeout);
                observacionesTimeout = setTimeout(function() {
                    $.ajax({
                        url: 'contactos.php',
                        method: 'POST',
                        data: {
                            action: 'actualizar_observaciones',
                            contacto_id: contactoId,
                            observaciones: nuevasObservaciones
                        },
                        success: function(response) {
                            if (!response.success) {
                                alert('Error: ' + response.message);
                            }
                        },
                        error: function() {
                            alert('Error de conexión');
                        }
                    });
                }, 1000);
            });
            <?php endif; ?>
            
            <?php if ($has_historial_table): ?>
            // Historial de cambios
            $('.historial-btn').click(function() {
                const contactoId = $(this).data('contacto-id');
                
                $.ajax({
                    url: 'contactos.php',
                    method: 'POST',
                    data: {
                        action: 'obtener_historial',
                        contacto_id: contactoId
                    },
                    success: function(response) {
                        if (response.success) {
                            let html = '';
                            if (response.historial.length > 0) {
                                html = '<div class="timeline">';
                                response.historial.forEach(function(cambio) {
                                    html += '<div class="timeline-item mb-3">';
                                    html += '<div class="card">';
                                    html += '<div class="card-body">';
                                    html += '<h6 class="card-title">' + cambio.campo_modificado + '</h6>';
                                    html += '<p class="card-text">';
                                    html += '<strong>Anterior:</strong> ' + (cambio.valor_anterior || 'N/A') + '<br>';
                                    html += '<strong>Nuevo:</strong> ' + (cambio.valor_nuevo || 'N/A');
                                    html += '</p>';
                                    html += '<small class="text-muted">';
                                    html += 'Por ' + cambio.usuario_nombre + ' el ' + new Date(cambio.fecha_cambio).toLocaleString();
                                    html += '</small>';
                                    html += '</div></div></div>';
                                });
                                html += '</div>';
                            } else {
                                html = '<p class="text-muted text-center">No hay cambios registrados</p>';
                            }
                            $('#historialContent').html(html);
                            $('#historialModal').modal('show');
                        } else {
                            alert('Error: ' + response.message);
                        }
                    },
                    error: function() {
                        alert('Error de conexión');
                    }
                });
            });
            <?php endif; ?>
            
            // Gestión de etiquetas
            <?php if ($has_etiquetas_table): ?>
            $('#etiquetasModal').on('show.bs.modal', function (event) {
                const button = $(event.relatedTarget);
                const contactoId = button.data('contacto-id');
                
                $.ajax({
                    url: 'contactos.php',
                    method: 'POST',
                    data: {
                        action: 'obtener_etiquetas_contacto',
                        contacto_id: contactoId
                    },
                    success: function(response) {
                        if (response.success) {
                            let html = '<div class="etiquetas-list">';
                            response.etiquetas.forEach(function(etiqueta) {
                                const checked = response.contacto_etiquetas.includes(etiqueta.id) ? 'checked' : '';
                                html += '<div class="form-check mb-2">';
                                html += '<input class="form-check-input etiqueta-checkbox" type="checkbox" value="' + etiqueta.id + '" id="etiq_' + etiqueta.id + '" ' + checked + '>';
                                html += '<label class="form-check-label" for="etiq_' + etiqueta.id + '">';
                                html += '<span class="badge me-2" style="background-color: ' + etiqueta.color + '">' + etiqueta.nombre + '</span>';
                                html += '</label>';
                                html += '</div>';
                            });
                            html += '</div>';
                            $('#etiquetasContent').html(html);
                            $('#guardarEtiquetas').data('contacto-id', contactoId);
                        } else {
                            $('#etiquetasContent').html('<p class="text-danger">Error: ' + response.message + '</p>');
                        }
                    },
                    error: function() {
                        $('#etiquetasContent').html('<p class="text-danger">Error de conexión</p>');
                    }
                });
            });
            
            $('#guardarEtiquetas').click(function() {
                const contactoId = $(this).data('contacto-id');
                const etiquetasSeleccionadas = [];
                $('.etiqueta-checkbox:checked').each(function() {
                    etiquetasSeleccionadas.push($(this).val());
                });
                
                $.ajax({
                    url: 'contactos.php',
                    method: 'POST',
                    data: {
                        action: 'actualizar_etiquetas_contacto',
                        contacto_id: contactoId,
                        etiquetas: etiquetasSeleccionadas
                    },
                    success: function(response) {
                        if (response.success) {
                            $('#etiquetasModal').modal('hide');
                            showAlert('Etiquetas actualizadas correctamente', 'success');
                        } else {
                            showAlert('Error: ' + response.message, 'danger');
                        }
                    },
                    error: function() {
                        showAlert('Error de conexión', 'danger');
                    }
                });
            });
             <?php endif; ?>
             
             // Botón Editar
             $('.editar-btn').click(function() {
                 const contactoId = $(this).data('contacto-id');
                 
                 $.ajax({
                     url: 'contactos.php',
                     method: 'POST',
                     data: {
                         action: 'obtener_contacto',
                         contacto_id: contactoId
                     },
                     success: function(response) {
                         if (response.success) {
                             const contacto = response.contacto;
                             $('#edit_contacto_id').val(contacto.id);
                             $('#edit_nombres_apellidos').val(contacto.nombres_apellidos);
                             $('#edit_celular').val(contacto.celular);
                             $('#edit_ciudad').val(contacto.ciudad);
                             $('#edit_fecha_registro').val(contacto.fecha_registro);
                             <?php if ($has_estado): ?>
                             $('#edit_estado').val(contacto.estado);
                             <?php endif; ?>
                             <?php if ($has_observaciones): ?>
                             $('#edit_observaciones').val(contacto.observaciones);
                             <?php endif; ?>
                         } else {
                             showAlert('Error: ' + response.message, 'danger');
                         }
                     },
                     error: function() {
                         showAlert('Error de conexión', 'danger');
                     }
                 });
             });
             
             // Formulario de editar contacto
             $('#editarContactoForm').submit(function(e) {
                 e.preventDefault();
                 
                 $.ajax({
                     url: 'contactos.php',
                     method: 'POST',
                     data: $(this).serialize() + '&action=actualizar_contacto',
                     success: function(response) {
                         if (response.success) {
                             $('#editarContactoModal').modal('hide');
                             showAlert('Contacto actualizado correctamente', 'success');
                             setTimeout(() => location.reload(), 1500);
                         } else {
                             showAlert('Error: ' + response.message, 'danger');
                         }
                     },
                     error: function() {
                         showAlert('Error de conexión', 'danger');
                     }
                 });
             });
             
             // Botón Separó
             $('.separo-btn').click(function() {
                 const contactoId = $(this).data('contacto-id');
                 
                 if (confirm('¿Confirmar que el contacto separó?')) {
                     $.ajax({
                         url: 'contactos.php',
                         method: 'POST',
                         data: {
                             action: 'marcar_separo',
                             contacto_id: contactoId
                         },
                         success: function(response) {
                             if (response.success) {
                                 showAlert('Contacto marcado como "Separó" correctamente', 'success');
                                 setTimeout(() => location.reload(), 1500);
                             } else {
                                 showAlert('Error: ' + response.message, 'danger');
                             }
                         },
                         error: function() {
                             showAlert('Error de conexión', 'danger');
                         }
                     });
                 }
             });
             
             // Botón Compró
             $('.compro-btn').click(function() {
                 const contactoId = $(this).data('contacto-id');
                 
                 if (confirm('¿Confirmar que el contacto compró?')) {
                     $.ajax({
                         url: 'contactos.php',
                         method: 'POST',
                         data: {
                             action: 'marcar_compro',
                             contacto_id: contactoId
                         },
                         success: function(response) {
                             if (response.success) {
                                 showAlert('Contacto marcado como "Compró" correctamente', 'success');
                                 setTimeout(() => location.reload(), 1500);
                             } else {
                                 showAlert('Error: ' + response.message, 'danger');
                             }
                         },
                         error: function() {
                             showAlert('Error de conexión', 'danger');
                         }
                     });
                 }
             });
         });
    </script>

    <!-- Modal Etiquetas -->
    <?php if ($has_etiquetas_table): ?>
    <div class="modal fade" id="etiquetasModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="fas fa-tags"></i> Gestionar Etiquetas</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="etiquetasContent">
                        <div class="text-center">
                            <div class="spinner-border" role="status">
                                <span class="visually-hidden">Cargando...</span>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                    <button type="button" class="btn btn-primary" id="guardarEtiquetas">Guardar Cambios</button>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

</body>
</html>