<?php
// Archivo de configuración global para el sistema CRM

// Función para obtener la configuración actual
function getSystemSettings() {
    $settingsFile = __DIR__ . '/assets/system_settings.json';
    
    // Configuración por defecto
    $defaultSettings = [
        'primary_color' => '#dc2626',
        'secondary_color' => '#b91c1c',
        'accent_color' => '#991b1b',
        'success_color' => '#10b981',
        'warning_color' => '#f59e0b',
        'danger_color' => '#ef4444',
        'company_name' => 'Ecoterrenos',
        'logo_path' => '',
        'logo_size' => '48',
        'font_size' => 'medium',
        'compact_mode' => false
    ];
    
    // Auto-detectar logo en la carpeta assets/logos
    $logoDir = 'assets/logos/';
    $logoExtensions = ['svg', 'png', 'jpg', 'jpeg', 'gif'];
    
    if (is_dir($logoDir)) {
         foreach ($logoExtensions as $ext) {
             $logoFile = $logoDir . 'logo.' . $ext;
             if (file_exists($logoFile)) {
                 $defaultSettings['logo_path'] = $logoFile;
                 break;
             }
         }
     }
    
    if (file_exists($settingsFile)) {
        $settings = json_decode(file_get_contents($settingsFile), true);
        return array_merge($defaultSettings, $settings);
    }
    
    return $defaultSettings;
}

// Función para guardar la configuración
function saveSystemSettings($settings) {
    $settingsFile = __DIR__ . '/assets/system_settings.json';
    $assetsDir = dirname($settingsFile);
    
    if (!is_dir($assetsDir)) {
        mkdir($assetsDir, 0755, true);
    }
    
    return file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT));
}

// Función para subir logo
function uploadLogo($file) {
    $uploadDir = __DIR__ . '/assets/logos/';
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/svg+xml'];
    $maxSize = 2 * 1024 * 1024; // 2MB
    
    if (!in_array($file['type'], $allowedTypes)) {
        return ['success' => false, 'message' => 'Tipo de archivo no permitido. Use JPG, PNG, GIF o SVG.'];
    }
    
    if ($file['size'] > $maxSize) {
        return ['success' => false, 'message' => 'El archivo es muy grande. Máximo 2MB.'];
    }
    
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = 'logo_' . time() . '.' . $extension;
    $filepath = $uploadDir . $filename;
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return ['success' => true, 'path' => 'assets/logos/' . $filename];
    }
    
    return ['success' => false, 'message' => 'Error al subir el archivo.'];
}

// Función para generar CSS dinámico
function generateDynamicCSS($settings = null) {
    if ($settings === null) {
        $settings = getSystemSettings();
    }
    
    $css = ":root {\n";
    $css .= "    --primary-color: {$settings['primary_color']};\n";
    $css .= "    --secondary-color: {$settings['secondary_color']};\n";
    $css .= "    --accent-color: {$settings['accent_color']};\n";
    $css .= "    --success-color: {$settings['success_color']};\n";
    $css .= "    --warning-color: {$settings['warning_color']};\n";
    $css .= "    --danger-color: {$settings['danger_color']};\n";
    
    if (!empty($settings['logo_path'])) {
        $css .= "    --logo-url: url('{$settings['logo_path']}');\n";
    } else {
        // Logo por defecto (SVG)
        $css .= "    --logo-url: url('data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMzIiIGhlaWdodD0iMzIiIHZpZXdCb3g9IjAgMCAzMiAzMiIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHJlY3Qgd2lkdGg9IjMyIiBoZWlnaHQ9IjMyIiByeD0iOCIgZmlsbD0iIzYzNjZmMSIvPgo8cGF0aCBkPSJNOCAxMmg0djhoLTR2LTh6bTYgMGg0djhoLTR2LTh6bTYgMGg0djhoLTR2LTh6IiBmaWxsPSJ3aGl0ZSIvPgo8L3N2Zz4K');\n";
    }
    
    // Tamaño del logo
    $logoSize = $settings['logo_size'] ?? '48';
    $css .= "    --logo-size: {$logoSize}px;\n";
    
    $css .= "}\n";
    
    // Estilos del sidebar con degradado rojo
    $css .= "
.sidebar {
    background: linear-gradient(135deg, #dc2626 0%, #b91c1c 50%, #991b1b 100%) !important;
    box-shadow: 2px 0 10px rgba(220, 38, 38, 0.3);
}

.sidebar-header {
    background: linear-gradient(135deg, #b91c1c 0%, #991b1b 50%, #7f1d1d 100%) !important;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.sidebar .nav-link {
    color: rgba(255, 255, 255, 0.9) !important;
    transition: all 0.3s ease;
}

.sidebar .nav-link:hover {
    background: rgba(255, 255, 255, 0.1) !important;
    color: white !important;
    transform: translateX(5px);
}

.sidebar .nav-link.active {
    background: rgba(255, 255, 255, 0.2) !important;
    color: white !important;
    border-left: 3px solid white;
}

@media (max-width: 768px) {
    .sidebar {
        background: linear-gradient(180deg, #dc2626 0%, #b91c1c 50%, #991b1b 100%) !important;
    }
}
";
    
    // Estilos específicos para el logo
     $css .= "
/* Estilos del logo */
.sidebar-header::before {
    content: '';
    display: inline-block;
    width: var(--logo-size);
    height: var(--logo-size);
    background: var(--logo-url);
    background-size: contain;
    background-repeat: no-repeat;
    background-position: center;
    margin-right: 12px;
    vertical-align: middle;
    border-radius: 8px;
    transition: all 0.3s ease;
    filter: drop-shadow(0 2px 4px rgba(0, 0, 0, 0.3));
}

.sidebar-header h2, .sidebar-header h4 {
    display: inline-block;
    vertical-align: middle;
    margin: 0;
    font-size: 1.25rem;
    font-weight: 600;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
}

.sidebar-header {
    display: flex;
    align-items: center;
    padding: 1.5rem 1rem;
    border-bottom: 1px solid rgba(255,255,255,0.1);
}

/* Ajustes responsivos para el logo */
@media (max-width: 768px) {
    .sidebar-header::before {
        width: 40px;
        height: 40px;
        margin-right: 8px;
    }
    .sidebar-header h2, .sidebar-header h4 {
        font-size: 1.1rem;
    }
}
";
    
    // Estilos para tamaño de fuente
    if ($settings['font_size'] === 'small') {
        $css .= "body { font-size: 0.875rem; }\n";
    } elseif ($settings['font_size'] === 'large') {
        $css .= "body { font-size: 1.125rem; }\n";
    }
    
    // Estilos para modo compacto
    if ($settings['compact_mode']) {
        $css .= ".sidebar { width: 60px; }\n";
        $css .= ".main-content { margin-left: 60px; }\n";
        $css .= ".card { padding: 1rem; }\n";
        $css .= ".sidebar-header::before { width: 32px; height: 32px; margin-right: 0; }\n";
        $css .= ".sidebar-header h2, .sidebar-header h4 { display: none; }\n";
    }
    
    return $css;
}

// Procesar solicitudes AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'save_settings':
            $settings = [
                'primary_color' => $_POST['primary_color'] ?? '#6366f1',
                'secondary_color' => $_POST['secondary_color'] ?? '#8b5cf6',
                'accent_color' => $_POST['accent_color'] ?? '#06b6d4',
                'success_color' => $_POST['success_color'] ?? '#10b981',
                'warning_color' => $_POST['warning_color'] ?? '#f59e0b',
                'danger_color' => $_POST['danger_color'] ?? '#ef4444',
                'company_name' => $_POST['company_name'] ?? 'Mi Empresa',
                'logo_size' => $_POST['logo_size'] ?? '48',
                'font_size' => $_POST['font_size'] ?? 'medium',
                'compact_mode' => isset($_POST['compact_mode'])
            ];
            
            // Mantener el logo actual si no se sube uno nuevo
            $currentSettings = getSystemSettings();
            $settings['logo_path'] = $currentSettings['logo_path'];
            
            if (saveSystemSettings($settings)) {
                echo json_encode(['success' => true, 'message' => 'Configuración guardada exitosamente']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Error al guardar la configuración']);
            }
            break;
            
        case 'upload_logo':
            if (isset($_FILES['logo'])) {
                $result = uploadLogo($_FILES['logo']);
                if ($result['success']) {
                    // Actualizar la configuración con el nuevo logo
                    $settings = getSystemSettings();
                    $settings['logo_path'] = $result['path'];
                    saveSystemSettings($settings);
                }
                echo json_encode($result);
            } else {
                echo json_encode(['success' => false, 'message' => 'No se recibió ningún archivo']);
            }
            break;
            
        case 'get_settings':
            echo json_encode(getSystemSettings());
            break;
            
        case 'reset_settings':
            $defaultSettings = [
                'primary_color' => '#6366f1',
                'secondary_color' => '#8b5cf6',
                'accent_color' => '#06b6d4',
                'success_color' => '#10b981',
                'warning_color' => '#f59e0b',
                'danger_color' => '#ef4444',
                'logo_path' => '',
                'company_name' => 'Mi Empresa',
                'font_size' => 'medium',
                'compact_mode' => false
            ];
            
            if (saveSystemSettings($defaultSettings)) {
                echo json_encode(['success' => true, 'message' => 'Configuración restablecida']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Error al restablecer la configuración']);
            }
            break;
    }
    exit;
}
?>