<?php
require_once 'config.php';
requireAdmin();

$database = new Database();
$db = $database->getConnection();

// Manejar acciones AJAX para gestión de etiquetas
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'get_etiquetas':
            // Obtener etiquetas desde la tabla
            $query = "SELECT * FROM etiquetas ORDER BY nombre";
            $stmt = $db->prepare($query);
            $stmt->execute();
            $etiquetas = $stmt->fetchAll();
            
            echo json_encode(['success' => true, 'etiquetas' => $etiquetas]);
            exit;
            
        case 'add_etiqueta':
            try {
                $nombre = trim($_POST['nombre']);
                $color = $_POST['color'] ?? '#6366f1';
                
                if (empty($nombre)) {
                    throw new Exception('El nombre de la etiqueta es requerido');
                }
                
                // Verificar si ya existe
                $query_check = "SELECT COUNT(*) FROM etiquetas WHERE nombre = :nombre";
                $stmt_check = $db->prepare($query_check);
                $stmt_check->bindParam(':nombre', $nombre);
                $stmt_check->execute();
                
                if ($stmt_check->fetchColumn() > 0) {
                    throw new Exception('Ya existe una etiqueta con ese nombre');
                }
                
                // Insertar nueva etiqueta
                $query = "INSERT INTO etiquetas (nombre, color) VALUES (:nombre, :color)";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':nombre', $nombre);
                $stmt->bindParam(':color', $color);
                $stmt->execute();
                
                echo json_encode(['success' => true, 'message' => 'Etiqueta agregada correctamente']);
            } catch (Exception $e) {
                echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
            }
            exit;
            
        case 'update_etiqueta':
            try {
                $id = $_POST['id'];
                $nombre = trim($_POST['nombre']);
                $color = $_POST['color'] ?? '#6366f1';
                
                if (empty($nombre)) {
                    throw new Exception('El nombre de la etiqueta es requerido');
                }
                
                // Verificar si ya existe otro con el mismo nombre
                $query_check = "SELECT COUNT(*) FROM etiquetas WHERE nombre = :nombre AND id != :id";
                $stmt_check = $db->prepare($query_check);
                $stmt_check->bindParam(':nombre', $nombre);
                $stmt_check->bindParam(':id', $id);
                $stmt_check->execute();
                
                if ($stmt_check->fetchColumn() > 0) {
                    throw new Exception('Ya existe una etiqueta con ese nombre');
                }
                
                // Actualizar etiqueta
                $query = "UPDATE etiquetas SET nombre = :nombre, color = :color WHERE id = :id";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':nombre', $nombre);
                $stmt->bindParam(':color', $color);
                $stmt->bindParam(':id', $id);
                $stmt->execute();
                
                echo json_encode(['success' => true, 'message' => 'Etiqueta actualizada correctamente']);
            } catch (Exception $e) {
                echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
            }
            exit;
            
        case 'delete_etiqueta':
            try {
                $id = $_POST['id'];
                
                // Eliminar etiqueta
                $query = "DELETE FROM etiquetas WHERE id = :id";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':id', $id);
                $stmt->execute();
                
                echo json_encode(['success' => true, 'message' => 'Etiqueta eliminada correctamente']);
            } catch (Exception $e) {
                echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
            }
            exit;
    }
}

// Obtener estadísticas generales
$query_stats = "SELECT 
    (SELECT COUNT(*) FROM usuarios WHERE tipo = 'usuario') as total_usuarios,
    (SELECT COUNT(*) FROM contactos) as total_contactos,
    (SELECT COUNT(*) FROM contactos WHERE DATE(fecha_registro) = CURRENT_DATE()) as contactos_hoy,
    (SELECT COUNT(*) FROM contactos WHERE MONTH(fecha_registro) = MONTH(CURRENT_DATE()) AND YEAR(fecha_registro) = YEAR(CURRENT_DATE())) as contactos_mes";
$stmt_stats = $db->prepare($query_stats);
$stmt_stats->execute();
$stats = $stmt_stats->fetch();

// Obtener estadísticas por usuario
$query_usuarios = "SELECT 
    u.id,
    u.username,
    u.email,
    u.fecha_registro,
    COUNT(c.id) as total_contactos,
    COUNT(CASE WHEN DATE(c.fecha_registro) = CURRENT_DATE() THEN 1 END) as contactos_hoy,
    COUNT(CASE WHEN MONTH(c.fecha_registro) = MONTH(CURRENT_DATE()) AND YEAR(c.fecha_registro) = YEAR(CURRENT_DATE()) THEN 1 END) as contactos_mes
    FROM usuarios u
    LEFT JOIN contactos c ON u.id = c.usuario_id
    WHERE u.tipo = 'usuario'
    GROUP BY u.id, u.username, u.email, u.fecha_registro
    ORDER BY total_contactos DESC";
$stmt_usuarios = $db->prepare($query_usuarios);
$stmt_usuarios->execute();
$usuarios_stats = $stmt_usuarios->fetchAll();

// Obtener datos para gráfico de contactos por mes
$query_grafico_mes = "SELECT 
    MONTH(fecha_registro) as mes,
    COUNT(*) as cantidad
    FROM contactos 
    WHERE YEAR(fecha_registro) = YEAR(CURRENT_DATE())
    GROUP BY MONTH(fecha_registro)
    ORDER BY mes";
$stmt_grafico_mes = $db->prepare($query_grafico_mes);
$stmt_grafico_mes->execute();
$datos_mes = $stmt_grafico_mes->fetchAll();

// Obtener datos para gráfico de contactos por usuario (top 10)
$query_grafico_usuarios = "SELECT 
    u.username,
    COUNT(c.id) as cantidad
    FROM usuarios u
    LEFT JOIN contactos c ON u.id = c.usuario_id
    WHERE u.tipo = 'usuario'
    GROUP BY u.id, u.username
    ORDER BY cantidad DESC
    LIMIT 10";
$stmt_grafico_usuarios = $db->prepare($query_grafico_usuarios);
$stmt_grafico_usuarios->execute();
$datos_usuarios = $stmt_grafico_usuarios->fetchAll();

// Preparar datos para gráficos
$meses = ['Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun', 'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'];
$datos_meses_grafico = array_fill(0, 12, 0);
foreach ($datos_mes as $dato) {
    $datos_meses_grafico[$dato['mes'] - 1] = $dato['cantidad'];
}

$nombres_usuarios = [];
$cantidades_usuarios = [];
foreach ($datos_usuarios as $dato) {
    $nombres_usuarios[] = $dato['username'];
    $cantidades_usuarios[] = $dato['cantidad'];
}

// Obtener ciudades más populares
$query_ciudades = "SELECT ciudad, COUNT(*) as cantidad 
                  FROM contactos 
                  GROUP BY ciudad 
                  ORDER BY cantidad DESC 
                  LIMIT 10";
$stmt_ciudades = $db->prepare($query_ciudades);
$stmt_ciudades->execute();
$ciudades_populares = $stmt_ciudades->fetchAll();
?>

<?php
require_once 'settings.php';
$systemSettings = getSystemSettings();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Panel de Administración - <?php echo SITE_NAME; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="dynamic_styles.php" rel="stylesheet" type="text/css">
    <style>
        :root {
            --primary-color: #dc2626;
            --secondary-color: #b91c1c;
            --accent-color: #f87171;
            --success-color: #10b981;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --dark-color: #1f2937;
            --light-color: #f9fafb;
            --border-color: #e5e7eb;
            --text-primary: #111827;
            --text-secondary: #6b7280;
            --header-height: 70px;
        }
        
        * {
            font-family: 'Poppins', sans-serif;
            box-sizing: border-box;
        }
        
        body {
            background: var(--light-color);
            color: var(--text-primary);
            min-height: 100vh;
            margin: 0;
            padding-top: var(--header-height);
        }
        
        /* Estilos específicos de admin */
        
        .main-content {
            width: 100%;
            min-height: calc(100vh - var(--header-height));
            padding: 0;
        }
        
        .content-area {
            padding: 2rem;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 2rem;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
            border: none;
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-icon {
            width: 60px;
            height: 60px;
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            color: white;
            margin-bottom: 1rem;
        }
        
        .stat-number {
            font-size: 2.5rem;
            font-weight: 700;
            color: #2c3e50;
            margin: 0;
        }
        
        .stat-label {
            color: #7f8c8d;
            font-weight: 500;
            margin: 0;
        }
        
        .chart-container {
            background: white;
            border-radius: 15px;
            padding: 2rem;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
            margin-bottom: 2rem;
            min-height: 300px;
            max-height: 400px;
        }
        
        .no-data-message {
            text-align: center;
            color: #7f8c8d;
            font-style: italic;
            padding: 3rem;
        }
        
        .chart-wrapper {
            position: relative;
            height: 250px;
        }
        
        .table-container {
            background: white;
            border-radius: 15px;
            padding: 2rem;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }
        
        .table thead th {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            color: white;
            border: none;
            font-weight: 600;
        }
        
        .progress {
            height: 8px;
            border-radius: 10px;
        }
        
        .footer {
            background: white;
            padding: 1rem 2rem;
            text-align: center;
            color: #7f8c8d;
            border-top: 1px solid #e9ecef;
            margin-top: 2rem;
        }
        
        .footer i {
            color: #e74c3c;
        }
        
        /* Media Queries */
        @media (max-width: 768px) {
            .top-header {
                padding: 0 1rem;
                flex-wrap: wrap;
            }

            .nav-menu {
                position: absolute;
                top: 100%;
                left: 0;
                right: 0;
                background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
                flex-direction: column;
                gap: 0;
                display: none;
                box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
            }

            .nav-menu.show {
                display: flex;
            }

            .nav-menu a {
                padding: 1rem;
                border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            }

            .nav-menu a:hover,
            .nav-menu a.active {
                background: rgba(255, 255, 255, 0.2);
            }

            .mobile-menu-btn {
                display: block;
            }

            .content-area {
                padding: 1rem;
            }
        }

        @media (max-width: 480px) {
            .logo-container h4 {
                display: none;
            }

            .user-dropdown span {
                display: none;
            }
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    
    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <!-- Content Area -->
        <div class="content-area" style="padding: 2rem;">
            <!-- Statistics Cards -->
            <div class="row mb-4">
                <div class="col-lg-3 col-md-6 mb-4">
                    <div class="stat-card">
                        <div class="stat-icon" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                            <i class="fas fa-users"></i>
                        </div>
                        <h3 class="stat-number"><?php echo number_format($stats['total_usuarios']); ?></h3>
                        <p class="stat-label">Total Usuarios</p>
                    </div>
                </div>
                
                <div class="col-lg-3 col-md-6 mb-4">
                    <div class="stat-card">
                        <div class="stat-icon" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);">
                            <i class="fas fa-address-book"></i>
                        </div>
                        <h3 class="stat-number"><?php echo number_format($stats['total_contactos']); ?></h3>
                        <p class="stat-label">Total Contactos</p>
                    </div>
                </div>
                
                <div class="col-lg-3 col-md-6 mb-4">
                    <div class="stat-card">
                        <div class="stat-icon" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);">
                            <i class="fas fa-calendar-day"></i>
                        </div>
                        <h3 class="stat-number"><?php echo number_format($stats['contactos_hoy']); ?></h3>
                        <p class="stat-label">Contactos Hoy</p>
                    </div>
                </div>
                
                <div class="col-lg-3 col-md-6 mb-4">
                    <div class="stat-card">
                        <div class="stat-icon" style="background: linear-gradient(135deg, #fa709a 0%, #fee140 100%);">
                            <i class="fas fa-calendar-alt"></i>
                        </div>
                        <h3 class="stat-number"><?php echo number_format($stats['contactos_mes']); ?></h3>
                        <p class="stat-label">Contactos Este Mes</p>
                    </div>
                </div>
            </div>
            
            <!-- Charts Row -->
            <div class="row mb-4">
                <div class="col-lg-8">
                    <div class="chart-container">
                        <h5 class="mb-4">
                            <i class="fas fa-chart-line me-2"></i>
                            Contactos por Mes (<?php echo date('Y'); ?>)
                        </h5>
                        <?php if (array_sum($datos_meses_grafico) > 0): ?>
                            <div class="chart-wrapper">
                                <canvas id="contactosMesChart"></canvas>
                            </div>
                        <?php else: ?>
                            <div class="no-data-message">
                                <i class="fas fa-chart-line fa-3x mb-3 text-muted"></i>
                                <p>No hay información de contactos para mostrar</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="col-lg-4">
                    <div class="chart-container">
                        <h5 class="mb-4">
                            <i class="fas fa-chart-pie me-2"></i>
                            Top Usuarios
                        </h5>
                        <?php if (!empty($nombres_usuarios) && array_sum($cantidades_usuarios) > 0): ?>
                            <div class="chart-wrapper">
                                <canvas id="usuariosChart"></canvas>
                            </div>
                        <?php else: ?>
                            <div class="no-data-message">
                                <i class="fas fa-users fa-3x mb-3 text-muted"></i>
                                <p>No hay información de usuarios para mostrar</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Users Statistics Table -->
            <div class="table-container mb-4">
                <h5 class="mb-4">
                    <i class="fas fa-users me-2"></i>
                    Estadísticas por Usuario
                </h5>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Usuario</th>
                                <th>Email</th>
                                <th>Fecha Registro</th>
                                <th>Total Contactos</th>
                                <th>Hoy</th>
                                <th>Este Mes</th>
                                <th>Progreso</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($usuarios_stats as $usuario): ?>
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="avatar-sm bg-primary rounded-circle d-flex align-items-center justify-content-center me-2">
                                            <i class="fas fa-user text-white"></i>
                                        </div>
                                        <strong><?php echo htmlspecialchars($usuario['username']); ?></strong>
                                    </div>
                                </td>
                                <td><?php echo htmlspecialchars($usuario['email']); ?></td>
                                <td><?php echo formatDate($usuario['fecha_registro']); ?></td>
                                <td>
                                    <span class="badge bg-primary fs-6"><?php echo $usuario['total_contactos']; ?></span>
                                </td>
                                <td>
                                    <span class="badge bg-info fs-6"><?php echo $usuario['contactos_hoy']; ?></span>
                                </td>
                                <td>
                                    <span class="badge bg-success fs-6"><?php echo $usuario['contactos_mes']; ?></span>
                                </td>
                                <td>
                                    <?php 
                                    $max_contactos = max(array_column($usuarios_stats, 'total_contactos'));
                                    $porcentaje = $max_contactos > 0 ? ($usuario['total_contactos'] / $max_contactos) * 100 : 0;
                                    ?>
                                    <div class="progress">
                                        <div class="progress-bar bg-gradient" role="progressbar" 
                                             style="width: <?php echo $porcentaje; ?>%; background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);"
                                             aria-valuenow="<?php echo $porcentaje; ?>" aria-valuemin="0" aria-valuemax="100">
                                        </div>
                                    </div>
                                    <small class="text-muted"><?php echo number_format($porcentaje, 1); ?>%</small>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <!-- Popular Cities -->
            <div class="table-container">
                <h5 class="mb-4">
                    <i class="fas fa-map-marker-alt me-2"></i>
                    Ciudades Más Populares
                </h5>
                <div class="row">
                    <?php foreach ($ciudades_populares as $index => $ciudad): ?>
                    <div class="col-md-6 col-lg-4 mb-3">
                        <div class="d-flex align-items-center p-3 bg-light rounded">
                            <div class="me-3">
                                <span class="badge bg-primary rounded-pill"><?php echo $index + 1; ?></span>
                            </div>
                            <div class="flex-grow-1">
                                <h6 class="mb-0"><?php echo htmlspecialchars($ciudad['ciudad']); ?></h6>
                                <small class="text-muted"><?php echo $ciudad['cantidad']; ?> contactos</small>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <!-- Gestión de Etiquetas -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="table-container">
                    <h5 class="mb-4">
                        <i class="fas fa-tags me-2"></i>
                        Gestión de Etiquetas
                    </h5>
                    
                    <!-- Formulario para agregar nueva etiqueta -->
                    <div class="card mb-4">
                        <div class="card-header">
                            <h6 class="mb-0"><i class="fas fa-plus-circle me-2"></i>Crear Nueva Etiqueta</h6>
                        </div>
                        <div class="card-body">
                            <div class="row g-3">
                                <div class="col-md-5">
                                    <label for="nueva-etiqueta-nombre" class="form-label">Nombre de la etiqueta</label>
                                    <input type="text" class="form-control" id="nueva-etiqueta-nombre" placeholder="Ej: Cliente VIP, Prospecto, etc.">
                                </div>
                                <div class="col-md-3">
                                    <label for="nueva-etiqueta-color" class="form-label">Color</label>
                                    <input type="color" class="form-control form-control-color" id="nueva-etiqueta-color" value="#6366f1">
                                </div>
                                <div class="col-md-4 d-flex align-items-end">
                                    <button type="button" class="btn btn-success w-100" id="add-etiqueta-btn">
                                        <i class="fas fa-plus"></i> Crear Etiqueta
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Lista de etiquetas existentes -->
                    <div class="card">
                        <div class="card-header">
                            <h6 class="mb-0"><i class="fas fa-list me-2"></i>Etiquetas Existentes</h6>
                        </div>
                        <div class="card-body">
                            <div id="etiquetas-list">
                                <div class="text-center py-4">
                                    <div class="spinner-border text-primary" role="status">
                                        <span class="visually-hidden">Cargando...</span>
                                    </div>
                                    <p class="mt-2 text-muted">Cargando etiquetas...</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Información adicional -->
                    <div class="row mt-4">
                        <div class="col-12">
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                <strong>Información:</strong> Las etiquetas permiten categorizar y organizar mejor los contactos. 
                                Puedes asignar múltiples etiquetas a cada contacto para facilitar la búsqueda y filtrado.
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Footer -->
        <footer class="footer">
            <p class="mb-0">Diseñado por Vortex Agencia <i class="fas fa-heart"></i></p>
        </footer>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Note: Mobile menu functionality is handled in header.php
        
        // Chart de contactos por mes
        <?php if (array_sum($datos_meses_grafico) > 0): ?>
        const ctxMes = document.getElementById('contactosMesChart');
        if (ctxMes) {
            const chartMes = new Chart(ctxMes.getContext('2d'), {
                type: 'line',
                data: {
                    labels: <?php echo json_encode($meses); ?>,
                    datasets: [{
                        label: 'Contactos',
                        data: <?php echo json_encode($datos_meses_grafico); ?>,
                        borderColor: 'rgba(102, 126, 234, 1)',
                        backgroundColor: 'rgba(102, 126, 234, 0.1)',
                        borderWidth: 3,
                        fill: true,
                        tension: 0.4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                stepSize: 1
                            }
                        }
                    }
                }
            });
        }
        <?php endif; ?>
        
        // Chart de usuarios
        <?php if (!empty($nombres_usuarios) && array_sum($cantidades_usuarios) > 0): ?>
        const ctxUsuarios = document.getElementById('usuariosChart');
        if (ctxUsuarios) {
            const chartUsuarios = new Chart(ctxUsuarios.getContext('2d'), {
                type: 'doughnut',
                data: {
                    labels: <?php echo json_encode($nombres_usuarios); ?>,
                    datasets: [{
                        data: <?php echo json_encode($cantidades_usuarios); ?>,
                        backgroundColor: [
                            '#667eea', '#764ba2', '#f093fb', '#f5576c',
                            '#4facfe', '#00f2fe', '#fa709a', '#fee140',
                            '#a8edea', '#fed6e3'
                        ],
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: {
                                padding: 15,
                                usePointStyle: true,
                                font: {
                                    size: 11
                                }
                            }
                        }
                    }
                }
            });
        }
        <?php endif; ?>
        
        // Close mobile menu on resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768 && navMenu) {
                navMenu.classList.remove('show');
            }
        });
        
        // Gestión de Etiquetas
        let etiquetas = [];
        
        // Cargar etiquetas al iniciar
        cargarEtiquetas();
        
        function cargarEtiquetas() {
            console.log('🔄 Iniciando carga de etiquetas...');
            
            fetch('admin.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'action=get_etiquetas'
            })
            .then(response => {
                console.log('📡 Respuesta recibida:', response.status);
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                console.log('📊 Datos recibidos:', data);
                if (data.success) {
                    console.log('✅ Éxito! Etiquetas encontradas:', data.etiquetas.length);
                    etiquetas = data.etiquetas;
                    mostrarEtiquetas(data.etiquetas);
                } else {
                    console.error('❌ Error en respuesta:', data.message);
                    const container = document.getElementById('etiquetas-list');
                    container.innerHTML = `
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            Error: ${data.message || 'Error desconocido'}
                        </div>
                    `;
                }
            })
            .catch(error => {
                console.error('💥 Error en fetch:', error);
                const container = document.getElementById('etiquetas-list');
                container.innerHTML = `
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Error de conexión: ${error.message}
                    </div>
                `;
            });
        }
        
        function mostrarEtiquetas(etiquetas) {
            const container = document.getElementById('etiquetas-list');
            container.innerHTML = '';
            
            if (etiquetas.length === 0) {
                container.innerHTML = `
                    <div class="text-center py-4">
                        <i class="fas fa-tags fa-3x text-muted mb-3"></i>
                        <h6 class="text-muted">No hay etiquetas creadas</h6>
                        <p class="text-muted small">Crea tu primera etiqueta usando el formulario de arriba</p>
                    </div>
                `;
                return;
            }
            
            etiquetas.forEach((etiqueta, index) => {
                const etiquetaDiv = document.createElement('div');
                etiquetaDiv.className = 'card mb-3 shadow-sm';
                etiquetaDiv.innerHTML = `
                    <div class="card-body">
                        <div class="row align-items-center">
                            <div class="col-md-1 text-center">
                                <div class="color-preview" style="width: 40px; height: 40px; background-color: ${etiqueta.color}; border-radius: 50%; border: 3px solid #fff; box-shadow: 0 2px 4px rgba(0,0,0,0.1); margin: 0 auto;"></div>
                            </div>
                            <div class="col-md-5">
                                <label class="form-label small text-muted">Nombre de la etiqueta</label>
                                <input type="text" class="form-control etiqueta-nombre" value="${etiqueta.nombre}" data-id="${etiqueta.id}" placeholder="Nombre de la etiqueta">
                            </div>
                            <div class="col-md-3">
                                <label class="form-label small text-muted">Color</label>
                                <input type="color" class="form-control form-control-color etiqueta-color" value="${etiqueta.color}" data-id="${etiqueta.id}">
                            </div>
                            <div class="col-md-3 text-end">
                                <div class="btn-group" role="group">
                                    <button class="btn btn-outline-success btn-sm" onclick="actualizarEtiqueta(${etiqueta.id})" title="Guardar cambios">
                                        <i class="fas fa-save"></i> Guardar
                                    </button>
                                    <button class="btn btn-outline-danger btn-sm" onclick="eliminarEtiqueta(${etiqueta.id})" title="Eliminar etiqueta">
                                        <i class="fas fa-trash"></i> Eliminar
                                    </button>
                                </div>
                            </div>
                        </div>
                        <div class="row mt-2">
                            <div class="col-12">
                                <small class="text-muted">
                                    <i class="fas fa-info-circle me-1"></i>
                                    Etiqueta #${index + 1} - Modifica el nombre o color y haz clic en "Guardar" para aplicar los cambios
                                </small>
                            </div>
                        </div>
                    </div>
                `;
                container.appendChild(etiquetaDiv);
            });
        }
        
        function agregarEtiqueta() {
            const nombre = document.getElementById('nueva-etiqueta-nombre').value.trim();
            const color = document.getElementById('nueva-etiqueta-color').value;
            const btn = document.getElementById('add-etiqueta-btn');
            
            if (!nombre) {
                alert('Por favor ingresa un nombre para la etiqueta');
                document.getElementById('nueva-etiqueta-nombre').focus();
                return;
            }
            
            // Deshabilitar botón durante la operación
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Creando...';
            
            fetch('admin.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=add_etiqueta&nombre=${encodeURIComponent(nombre)}&color=${encodeURIComponent(color)}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('nueva-etiqueta-nombre').value = '';
                    document.getElementById('nueva-etiqueta-color').value = '#6366f1';
                    cargarEtiquetas();
                    
                    // Mostrar mensaje de éxito
                    const alertDiv = document.createElement('div');
                    alertDiv.className = 'alert alert-success alert-dismissible fade show mt-3';
                    alertDiv.innerHTML = `
                        <i class="fas fa-check-circle me-2"></i>
                        <strong>¡Éxito!</strong> La etiqueta "${nombre}" ha sido creada correctamente.
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    `;
                    document.querySelector('.card-body').appendChild(alertDiv);
                    
                    // Remover alerta después de 5 segundos
                    setTimeout(() => {
                        if (alertDiv.parentNode) {
                            alertDiv.remove();
                        }
                    }, 5000);
                } else {
                    alert('Error: ' + (data.message || 'No se pudo crear la etiqueta'));
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al agregar la etiqueta. Por favor, inténtalo de nuevo.');
            })
            .finally(() => {
                // Rehabilitar botón
                btn.disabled = false;
                btn.innerHTML = '<i class="fas fa-plus"></i> Crear Etiqueta';
            });
        }
        
        function actualizarEtiqueta(id) {
            const nombreInput = document.querySelector(`input.etiqueta-nombre[data-id="${id}"]`);
            const colorInput = document.querySelector(`input.etiqueta-color[data-id="${id}"]`);
            const btn = document.querySelector(`button[onclick="actualizarEtiqueta(${id})"]`);
            
            const nombre = nombreInput.value.trim();
            const color = colorInput.value;
            
            if (!nombre) {
                alert('El nombre de la etiqueta no puede estar vacío');
                nombreInput.focus();
                return;
            }
            
            // Deshabilitar botón durante la operación
            const originalText = btn.innerHTML;
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Guardando...';
            
            fetch('admin.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=update_etiqueta&id=${id}&nombre=${encodeURIComponent(nombre)}&color=${encodeURIComponent(color)}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    cargarEtiquetas();
                    
                    // Mostrar mensaje de éxito temporal
                    btn.innerHTML = '<i class="fas fa-check"></i> ¡Guardado!';
                    btn.className = 'btn btn-success btn-sm';
                    
                    setTimeout(() => {
                        btn.innerHTML = originalText;
                        btn.className = 'btn btn-outline-success btn-sm';
                    }, 2000);
                } else {
                    alert('Error: ' + (data.message || 'No se pudo actualizar la etiqueta'));
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error al actualizar la etiqueta. Por favor, inténtalo de nuevo.');
            })
            .finally(() => {
                btn.disabled = false;
            });
        }
        
        function eliminarEtiqueta(id) {
            // Obtener el nombre de la etiqueta para el mensaje de confirmación
            const nombreInput = document.querySelector(`input.etiqueta-nombre[data-id="${id}"]`);
            const nombreEtiqueta = nombreInput ? nombreInput.value : 'esta etiqueta';
            
            if (confirm(`¿Estás seguro de que quieres eliminar "${nombreEtiqueta}"?\n\nEsta acción no se puede deshacer y la etiqueta se quitará de todos los contactos que la tengan asignada.`)) {
                const btn = document.querySelector(`button[onclick="eliminarEtiqueta(${id})"]`);
                
                // Deshabilitar botón durante la operación
                const originalText = btn.innerHTML;
                btn.disabled = true;
                btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Eliminando...';
                
                fetch('admin.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `action=delete_etiqueta&id=${id}`
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Mostrar mensaje de éxito
                        const alertDiv = document.createElement('div');
                        alertDiv.className = 'alert alert-warning alert-dismissible fade show mt-3';
                        alertDiv.innerHTML = `
                            <i class="fas fa-trash me-2"></i>
                            <strong>Eliminado:</strong> La etiqueta "${nombreEtiqueta}" ha sido eliminada correctamente.
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        `;
                        document.querySelector('#etiquetas-list').parentNode.appendChild(alertDiv);
                        
                        // Remover alerta después de 5 segundos
                        setTimeout(() => {
                            if (alertDiv.parentNode) {
                                alertDiv.remove();
                            }
                        }, 5000);
                        
                        cargarEtiquetas();
                    } else {
                        alert('Error: ' + (data.message || 'No se pudo eliminar la etiqueta'));
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error al eliminar la etiqueta. Por favor, inténtalo de nuevo.');
                })
                .finally(() => {
                    btn.disabled = false;
                    btn.innerHTML = originalText;
                });
            }
        }
        
        // Event listeners
        document.getElementById('add-etiqueta-btn').addEventListener('click', agregarEtiqueta);
        
        document.getElementById('nueva-etiqueta-nombre').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                agregarEtiqueta();
            }
        });
    </script>
</body>
</html>